<?php

namespace App\Http\Controllers\Contabilidad;

use Illuminate\Support\Facades\Log;
use App\Models\Parametros\ParEmpresa;
use App\Models\Contabilidad\ConImpuesto;
use Illuminate\Http\Request;
use App\Models\Contabilidad\ConImpuestoCuenta;
use App\Http\Controllers\Controller;

class ConImpuestoController extends Controller
{
    public function index()
    {
        return ConImpuesto::where('estado',1)->get();
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'codigo' => 'required|string|max:50',
            'nombre' => 'required|string|max:255',
            'grupo' => 'nullable|string|max:100',
            'modo' => 'nullable|string|max:100',
            'aplica_a' => 'nullable|string|max:100',
            'base_calculo' => 'nullable|string|max:100',
            'prioridad' => 'nullable|integer',
            'unidad_base' => 'nullable|string|max:50',
            'factor_unidad' => 'nullable|numeric',
            'con_sub_cuenta_id' => 'nullable',
        ]);

        // Normalizar con_sub_cuenta_id (puede venir como objeto/array del multiselect)
        $subCuentaId = null;
        if (isset($validated['con_sub_cuenta_id'])) {
            if (is_array($validated['con_sub_cuenta_id']) && isset($validated['con_sub_cuenta_id']['id'])) {
                $subCuentaId = $validated['con_sub_cuenta_id']['id'];
            } elseif (is_object($validated['con_sub_cuenta_id']) && isset($validated['con_sub_cuenta_id']->id)) {
                $subCuentaId = $validated['con_sub_cuenta_id']->id;
            } else {
                $subCuentaId = $validated['con_sub_cuenta_id'];
            }
        }

        $conImpuesto = ConImpuesto::create([
            'codigo' => $validated['codigo'],
            'nombre' => $validated['nombre'],
            'grupo' => $validated['grupo'] ?? null,
            'modo' => $validated['modo'] ?? null,
            'aplica_a' => $validated['aplica_a'] ?? null,
            'base_calculo' => $validated['base_calculo'] ?? null,
            'prioridad' => $validated['prioridad'] ?? null,
            'unidad_base' => $validated['unidad_base'] ?? null,
            'factor_unidad' => $validated['factor_unidad'] ?? null,
            'con_sub_cuenta_id' => $subCuentaId,
        ]);

        // Crear relaciones con todas las empresas
        $empresa_ids = ParEmpresa::pluck('id');
        foreach ($empresa_ids as $empresa_id) {
            ConImpuestoCuenta::create([
                'par_empresa_id' => $empresa_id,
                'con_impuesto_id' => $conImpuesto->id,
                'con_sub_cuenta_id' => $subCuentaId,
            ]);
        }

        return response()->json([
            'message' => 'Impuesto creado correctamente',
            'data' => $conImpuesto
        ], 201);
    }

    public function show(ConImpuesto $conImpuesto)
    {
        return response()->json($conImpuesto);
    }

    public function update(Request $request, ConImpuesto $conImpuesto)
    {
        $validated = $request->validate([
            'codigo' => 'required|string|max:50',
            'nombre' => 'required|string|max:255',
            'grupo' => 'nullable|string|max:100',
            'modo' => 'nullable|string|max:100',
            'aplica_a' => 'nullable|string|max:100',
            'base_calculo' => 'nullable|string|max:100',
            'prioridad' => 'nullable|integer',
            'unidad_base' => 'nullable|string|max:50',
            'factor_unidad' => 'nullable|numeric',
            'con_sub_cuenta_id' => 'nullable',
        ]);

        // Normalizar con_sub_cuenta_id
        $subCuentaId = null;
        if (isset($validated['con_sub_cuenta_id'])) {
            if (is_array($validated['con_sub_cuenta_id']) && isset($validated['con_sub_cuenta_id']['id'])) {
                $subCuentaId = $validated['con_sub_cuenta_id']['id'];
            } elseif (is_object($validated['con_sub_cuenta_id']) && isset($validated['con_sub_cuenta_id']->id)) {
                $subCuentaId = $validated['con_sub_cuenta_id']->id;
            } else {
                $subCuentaId = $validated['con_sub_cuenta_id'];
            }
        }

        $dataToUpdate = [
            'codigo' => $validated['codigo'],
            'nombre' => $validated['nombre'],
            'grupo' => $validated['grupo'] ?? null,
            'modo' => $validated['modo'] ?? null,
            'aplica_a' => $validated['aplica_a'] ?? null,
            'base_calculo' => $validated['base_calculo'] ?? null,
            'prioridad' => $validated['prioridad'] ?? null,
            'unidad_base' => $validated['unidad_base'] ?? null,
            'factor_unidad' => $validated['factor_unidad'] ?? null,
            'con_sub_cuenta_id' => $subCuentaId,
        ];

        $conImpuesto->update($dataToUpdate);

        // Actualizar también las cuentas asociadas si cambió con_sub_cuenta_id
        if ($subCuentaId !== null) {
            ConImpuestoCuenta::where('con_impuesto_id', $conImpuesto->id)
                ->update(['con_sub_cuenta_id' => $subCuentaId]);
        }

        return response()->json([
            'message' => 'Impuesto actualizado correctamente',
            'data' => $conImpuesto->fresh()
        ]);
    }

    public function destroy(ConImpuesto $conImpuesto)
    {
        $conImpuesto->estado = 0;
        $conImpuesto->save();
    }
}
