<?php

namespace App\Http\Controllers\Contabilidad;

use TCPDF;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Mpdf\Output\Destination;
use App\Models\Contabilidad\ConLibroFoliado;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;
use App\Models\Parametros\ParRepresentante;
use App\Models\Parametros\ParEmpresa as ParEmpresa;


class ConLibroFoliadoController extends Controller
{
    public function foliarAnualMayor(Request $request)
    {
        try {
            Log::info("📥 Datos recibidos en foliarAnualMayor", (array)$request->all());

            $empresa_id = $request->empresa_id;
            $f_inicial = $request->f_inicial;
            $tipo_libro = $request->tipo_libro;
            $por_periodo = $request->por_periodo;
            $t_periodo_id = $request->par_periodo_id ?? $request->t_periodo_id;
            $anio_id = $request->anio_id ?? $request->par_anio_id ?? $request->ano_id;
            $datos = $request->list ?? [];
            $user_id = auth()->id() ?? ($request->user_id ?? 0);

            $empresa = ParEmpresa::find($empresa_id);
            $anio = \App\Models\Parametros\ParAnio::find($anio_id);
            $representante = ParRepresentante::where('par_empresa_id', $empresa_id)->first();

            $empresa_nombre = $empresa->razon_social ?? $empresa->nombre ?? 'Empresa Desconocida';
            $empresa_rut = $empresa->rut ?? 'Rut Desconocido';
            $empresa_direccion = $empresa->direccion ?? 'Dirección Desconocida';
            $empresa_giro = $empresa->giro ?? 'Giro Desconocido';
            $anio_nro = $anio->nro ?? $anio_id;

            $representante_nombre = $representante
                ? implode(' ', array_filter([
                    $representante->p_nombre,
                    $representante->s_nombre,
                    $representante->a_paterno,
                    $representante->a_materno
                ]))
                : 'No registrado';
            $representante_rut = $representante->rut ?? null;

            // Obtener nombre del mes si es por periodo
            $mesNombre = null;
            if ($por_periodo == 1) {
                $periodo = DB::table('par_periodos')->where('id', $t_periodo_id)->first() ?? DB::table('t_periodos')->where('id', $t_periodo_id)->first();
                $mesNombre = $periodo
                    ? DB::table('meses')->where('id', $periodo->par_mese_id ?? $periodo->mese_id)->value('nombre')
                    : null;
            }

            $tituloLibro = $por_periodo == 1
                ? "LIBRO MAYOR PERIODO {$mesNombre} {$anio_nro}"
                : "LIBRO MAYOR AÑO {$anio_nro}";
            $tituloLibro = strtoupper($tituloLibro);

            $rutSanitizado = str_replace('.', '', $empresa_rut);
            $nombreArchivo = $por_periodo == 1
                ? "{$rutSanitizado}_Libro_Mayor_{$mesNombre}_{$anio_nro}.pdf"
                : "{$rutSanitizado}_Libro_Mayor_{$anio_nro}.pdf";

            $rutaRelativa = "libros-foliados/libro-mayor/{$nombreArchivo}";
            $rutaAbsoluta = storage_path("app/public/{$rutaRelativa}");

            Log::info("🧾 Generando PDF para archivo: $nombreArchivo");

            $html = view('pdfs.libro_mayor_tcpdf', [
                'datos' => $datos,
                'empresa_id' => $empresa_id,
                'anio_nro' => $anio_nro,
                'f_inicial' => $f_inicial,
                'empresa_nombre' => $empresa_nombre,
                'empresa_rut' => $empresa_rut,
                'empresa_direccion' => $empresa_direccion,
                'empresa_giro' => $empresa_giro,
                'representante_nombre' => $representante_nombre,
                'representante_rut' => $representante_rut,
                'tituloLibro' => $tituloLibro,
            ])->render();

            $pdf = new TCPDF('P', 'mm', 'LETTER', true, 'UTF-8', false);
            $pdf->SetCreator('Sistema');
            $pdf->SetAuthor('Contabilidad');
            $pdf->SetTitle($tituloLibro);
            $pdf->SetMargins(10, 40, 10);
            $pdf->SetHeaderMargin(8);
            $pdf->SetAutoPageBreak(true, 20);
            $pdf->setPrintHeader(true);
            $pdf->setPrintFooter(false);
            $pdf->SetHeaderData(
                '',
                0,
                '',
                "Empresa: {$empresa_nombre}\nRUT: {$empresa_rut}\nDirección: {$empresa_direccion}\nGiro: {$empresa_giro}\nRepresentante: {$representante_nombre}\nRUT Rep Leg: {$representante_rut}",
                [0, 0, 0],
                [0, 0, 0]
            );
            $pdf->setHeaderFont(['helvetica', '', 8]);

            $pdf->AddPage();
            $pdf->writeHTML($html, true, false, true, false, '');

            $pageCount = $pdf->getNumPages();
            $folio_final = $f_inicial + $pageCount - 1;

            for ($i = 1; $i <= $pageCount; $i++) {
                $pdf->setPage($i);
                $folio = $f_inicial + ($i - 1);
                $pdf->SetXY(180, 10);
                $pdf->SetFont('helvetica', '', 9);
                $pdf->Cell(25, 5, "Folio: $folio", 0, 0, 'R');
            }

            $rutaDirectorio = dirname($rutaAbsoluta);
            if (!file_exists($rutaDirectorio)) {
                mkdir($rutaDirectorio, 0775, true);
            }

            $pdf->Output($rutaAbsoluta, 'F');

            if (!file_exists($rutaAbsoluta)) {
                return response()->json([
                    'status' => 'error',
                    'mensaje' => 'El archivo PDF no se guardó correctamente en el disco.'
                ], 500);
            }

            // Guardar en tabla con_libro_foliado
            $libro = new ConLibroFoliado();
            $libro->par_empresa_id = $empresa_id;
            $libro->par_anio_id = $anio_id;
            $libro->tipo_libro = $tipo_libro;
            $libro->por_periodo = $por_periodo;
            $libro->par_periodo_id = $t_periodo_id;
            $libro->f_inicial = $f_inicial;
            $libro->f_final = $folio_final;
            $libro->url = Storage::url($rutaRelativa);
            $libro->user_id = $user_id;
            $libro->save();

            // Actualizar con_folio_libros
            DB::table('con_folio_libros')
                ->updateOrInsert(
                    ['par_empresa_id' => $empresa_id],
                    ['folio' => $folio_final]
                );

            // Actualizar par_periodo_empresas según por_periodo
            if ($por_periodo == 1) {
                DB::table('par_periodo_empresas')
                    ->where('par_empresa_id', $empresa_id)
                    ->where('par_periodo_id', $t_periodo_id)
                    ->update(['l_d_foliado' => 1]);
            } else {
                // Si es anual, marcar todos los periodos del año como foliados
                $periodos = DB::table('t_periodos')
                    ->where('par_anio_id', $anio_id)
                    ->pluck('id');

                DB::table('par_periodo_empresas')
                    ->where('par_empresa_id', $empresa_id)
                    ->whereIn('par_periodo_id', $periodos)
                    ->update(['l_d_foliado' => 1]);
            }

            return response()->json([
                'status' => 'ok',
                'folio_final' => $folio_final,
                'url' => asset($libro->url)
            ]);

        } catch (\Exception $e) {
            Log::error('❌ Error en foliarAnualMayor: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'mensaje' => 'Error al guardar el libro: ' . $e->getMessage()
            ], 500);
        }
    }

    public function filtroLDiarioMensual(Request $request)
    {
        $tipo = $request->tipo;
        $filtro = $request->filtro;
        $periodoId = $request->periodo;
        $anio = $request->anio;
        $anioAnterior = $anio - 1;

        // 1. Validar tipo_libro
        if (!in_array($tipo, ['1', '2', '3'])) {
            return response()->json(['status' => 'error', 'mensaje' => 'Tipo de libro no válido']);
        }

        // Validación adicional para tipo 2
        if ($tipo == '2') {
            $ldFoliado = DB::table('par_periodo_empresas')
                ->where('empresa_id', $filtro)
                ->where('par_periodo_id', $periodoId)
                ->value('l_d_foliado');

            if ($ldFoliado != 1) {
                return response()->json([
                    'status' => 'error',
                    'mensaje' => 'No tiene libro diario foliado para ese periodo'
                ]);
            }
        }

        // 2. Si el año ya está foliado completo
        $folioAnual = ConLibroFoliado::where('empresa_id', $filtro)
            ->where('par_anio_id', $anio)
            ->where('por_periodo', 0)
            ->where('tipo_libro', $tipo)
            ->where('estado', 1)
            ->exists();

        if ($folioAnual) {
            return response()->json(['status' => 'error', 'mensaje' => 'Año ya se encuentra foliado']);
        }

        // 3. Si ya existe el folio para el periodo actual
        $existePeriodo = ConLibroFoliado::where('empresa_id', $filtro)
            ->where('par_anio_id', $anio)
            ->where('par_periodo_id', $periodoId)
            ->where('por_periodo', 1)
            ->where('tipo_libro', $tipo)
            ->where('estado', 1)
            ->exists();

        if ($existePeriodo) {
            return response()->json(['status' => 'error', 'mensaje' => 'Periodo ya se encuentra foliado']);
        }

        // 4. Obtener los periodos del año anterior para la empresa
        $periodosAnioAnterior = DB::table('par_periodo_empresas as ppe')
            ->join('par_periodos as tp', 'ppe.par_periodo_id', '=', 'tp.id')
            ->where('ppe.empresa_id', $filtro)
            ->where('tp.par_anio_id', $anioAnterior)
            ->orderBy('tp.par_mese_id')
            ->pluck('ppe.par_periodo_id');

        $foliosAnteriores = ConLibroFoliado::where('empresa_id', $filtro)
            ->where('par_anio_id', $anioAnterior)
            ->whereIn('par_periodo_id', $periodosAnioAnterior)
            ->where('tipo_libro', $tipo)
            ->where('estado', 1)
            ->pluck('par_periodo_id');

        if ($periodosAnioAnterior->diff($foliosAnteriores)->isNotEmpty()) {
            return response()->json(['status' => 'warning', 'mensaje' => 'El año anterior no se encuentra foliado']);
        }

        // 5. Verificar si el periodo actual existe
        $periodoActual = DB::table('t_periodos')->where('id', $periodoId)->first();

        if (!$periodoActual) {
            return response()->json(['status' => 'error', 'mensaje' => 'Periodo no válido']);
        }

        // 6. Verificar que el periodo esté cerrado en par_periodo_empresas
        $estadoPeriodo = DB::table('par_periodo_empresas')
            ->where('empresa_id', $filtro)
            ->where('par_periodo_id', $periodoId)
            ->value('estado');

        if ($estadoPeriodo !== 0) {
            $nombreMes = DB::table('meses')->where('id', $periodoActual->par_mese_id ?? $periodoActual->mese_id)->value('nombre');

            return response()->json([
                'status' => 'error',
                'mensaje' => 'El periodo "' . ($nombreMes ?? 'Desconocido') . '" aún no se encuentra cerrado. No es posible foliar.'
            ]);
        }

        // 7. Verificar si el periodo anterior está foliado
        $periodoAnterior = DB::table('t_periodos')
            ->where('par_anio_id', $anio)
            ->where('par_mese_id', '<', $periodoActual->par_mese_id ?? $periodoActual->mese_id)
            ->orderByDesc('par_mese_id')
            ->first();

        if ($periodoAnterior) {
            $existeFolioAnterior = ConLibroFoliado::where('empresa_id', $filtro)
                ->where('par_anio_id', $anio)
                ->where('t_periodo_id', $periodoAnterior->id)
                ->where('tipo_libro', $tipo)
                ->where('estado', 1)
                ->exists();

            if (!$existeFolioAnterior) {
                // Buscar el último periodo foliado hacia atrás
                $periodosPrevios = DB::table('t_periodos')
                    ->where('par_anio_id', $anio)
                    ->where('par_mese_id', '<', $periodoActual->par_mese_id ?? $periodoActual->mese_id)
                    ->orderByDesc('par_mese_id')
                    ->get();

                foreach ($periodosPrevios as $p) {
                    $folio = ConLibroFoliado::where('empresa_id', $filtro)
                        ->where('par_anio_id', $anio)
                        ->where('par_periodo_id', $p->id)
                        ->where('tipo_libro', $tipo)
                        ->where('estado', 1)
                        ->first();

                    if ($folio) {
                        // Obtener el nombre del mes del último foliado
                        $mesNombre = DB::table('meses')->where('id', $p->par_mese_id ?? $p->mese_id)->value('nombre');

                        return response()->json([
                            'status' => 'warning',
                            'mensaje' => 'Último mes foliado',
                            'mes' => $mesNombre ?? 'Desconocido'
                        ]);
                    }
                }

                return response()->json(['status' => 'warning', 'mensaje' => 'No hay periodos anteriores foliados']);
            }
        }

        $folioMax = DB::table('con_folio_libros')
            ->where('empresa_id', $filtro)
            ->max('folio');

        $folioInicial = $folioMax ? $folioMax + 1 : 1;

        return response()->json([
            'status' => 'ok',
            'mensaje' => 'Ok para foliar',
            'folioInicial' => $folioInicial
        ]);
    }

    public function store(Request $request)
    {
        try {
            if ($request->hasFile('file')) {
                $file = $request->file('file');
                $nombre = Str::random(10) . '_' . $file->getClientOriginalName();

                $ruta = $file->storeAs('public/libros-foliados/libro-diario', $nombre);

                // Construir la URL pública
                $url = asset(Storage::url('libros-foliados/libro-diario/' . $nombre));

                // Guardar en base de datos
                $registro = new ConLibroFoliado();
                $registro->empresa_id = $request->empresa_id;
                $registro->tipo_libro = $request->tipo_libro;
                $registro->f_inicial = $request->f_inicial;
                $registro->f_final = $request->f_final;
                $registro->par_anio_id = $request->par_anio_id ?? $request->anio_id ?? $request->ano_id;
                $registro->url = $url;
                $registro->save();

                return response()->json([
                    'status' => 'ok',
                    'mensaje' => 'Archivo subido y datos registrados correctamente',
                    'url' => $url
                ]);
            } else {
                return response()->json([
                    'status' => 'error',
                    'mensaje' => 'No se encontró el archivo en la solicitud'
                ], 400);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'mensaje' => 'Error al guardar archivo: ' . $e->getMessage()
            ], 500);
        }
    }

    public function show(ConLibroFoliado $conLibroFoliado)
    {
        //
    }

    public function update(Request $request, ConLibroFoliado $conLibroFoliado)
    {
        //
    }

    public function destroy(ConLibroFoliado $conLibroFoliado)
    {
        //
    }
}
