<?php

namespace App\Http\Controllers\Contabilidad;
use App\Http\Controllers\Controller;

use App\Models\Parametros\ParEmpresa;
use App\Models\Parametros\ParPeriodo;
use App\Models\Pos\PosProveedor;
use App\Models\Contabilidad\ConSiiCompra;
use Illuminate\Http\Request;

use App\Models\Contabilidad\ConProveedorCuenta;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Crypt;
use Symfony\Component\Process\Process;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\Process\Exception\ProcessFailedException;




class ConSiiComprasController extends Controller
{
    public function index()
    {
        return ConSiiCompra::with(['ConSubCuenta', 'ConSubCuenta.cuenta.ConCategoriaCuenta.tipoCuenta'])->where('estado', 1)->get();
    }

    public function Filtro(Request $request)

    
    {
        $filtro = $request->filtro;
        $periodo = $request->periodo;
        return ConSiiCompra::with(['ConSubCuenta', 'ConSubCuenta.cuenta.ConCategoriaCuenta.tipoCuenta'])->where('par_empresa_id', $filtro)->where('par_periodo_id', $periodo)->orderBy('par_periodo_id')->where('estado',1)->get();
    }

     public function filtroRendicion(Request $request)

    
    {
        $empresa = $request->empresa;
        $rut_proveedor = $request->rut_proveedor;
        $folio = $request->folio;
        return ConSiiCompra::where('par_empresa_id', $empresa)->where('RUT_Proveedor', $rut_proveedor)->where('folio', $folio)->get();
    }


    public function subir(Request $request)
{
    $request->validate([
        'archivo' => 'required|file|mimes:csv,xml,zip',
    ]);

    $archivo = $request->file('archivo');
    $nombre = $archivo->getClientOriginalName();
    $archivo->storeAs('libros', $nombre, 'local');

    return response()->json(['mensaje' => '✅ Archivo subido']);
}

public function store(Request $request)
{
    // Validación MIME (como ya hicimos)
    $request->validate([
        'file' => 'required|file|mimetypes:text/csv,text/plain,application/pdf,text/xml,application/zip'
    ]);

    $archivo = $request->file('file');
    $nombre = $archivo->getClientOriginalName();

    // Ruta en storage/app/libros
    $rutaLibros = 'libros/' . $nombre;

    if (Storage::disk('local')->exists($rutaLibros)) {
        Storage::disk('local')->delete($rutaLibros);
    }

    // ✅ Guardar en storage/app/libros
    $archivo->storeAs('libros', $nombre, 'local');

    // 🚀 Ejecutar importación
    $importacion = app()->call([$this, 'importarComprasDesdeCSVManual'], ['nombreArchivo' => $nombre]);

    return response()->json([
        'mensaje' => '✅ Archivo subido e importado correctamente.',
        'resultado' => $importacion
    ]);
}
public function importarComprasDesdeCSVManual($nombreArchivo)
{
    // Ruta
    $path = storage_path('app/public/libros/' . $nombreArchivo);

    if (!file_exists($path)) {
        return response()->json(['error' => 'El archivo no existe: ' . $nombreArchivo], 404);
    }

    // Extraer RUT y periodo desde el nombre del archivo (validación estricta)
    if (preg_match('/RCV_COMPRA_REGISTRO_(.*?)_(.*?)\.csv/', $nombreArchivo, $matches) !== 1) {
        Log::warning("⚠️ Nombre de archivo inválido para importar compras: $nombreArchivo");
        return response()->json(['error' => 'Nombre de archivo inválido, formato esperado: RCV_COMPRA_REGISTRO_{RUT}_{PERIODO}.csv'], 400);
    }

    $rutEmpresaConDV = $matches[1];
    $periodo = $matches[2];
    $rutEmpresa = str_replace('-', '', $rutEmpresaConDV);

    $parEmpresa = ParEmpresa::where('rut', $rutEmpresa)->first();

    if (!$parEmpresa) {
        return response()->json(['error' => 'No se encontró la empresa con el RUT: ' . $rutEmpresa], 404);
    }

    $empresaId = $parEmpresa->id;
    // Buscar el período en la tabla ParPeriodo usando la columna mes_ano (valor extraído desde el nombre del archivo)
    $parPeriodo = ParPeriodo::where('mes_ano', $periodo)->first();

    if (!$parPeriodo) {
        return response()->json(['error' => 'No se encontró el período para el valor ' . $periodo], 404);
    }

    // Obtener el id del período
    $periodoId = $parPeriodo->id;


    // Leer CSV
    $file = fopen($path, 'r');
    $header = fgetcsv($file, 0, ';');

    // Crear un mapa de columnas basado en la cabecera
    $cabeceras = array_map('trim', $header);



    $registros = [];

    while (($row = fgetcsv($file, 0, ';')) !== false) {

        // Saltar si la fila tiene menos columnas
        if (count($row) < 5) continue;

        $rutProveedor = $this->buscarValor($cabeceras, $row, 'RUT Proveedor');
        $rutProveedorLimpio = str_replace('-', '', $rutProveedor);

        // Buscar proveedor por RUT
        $proveedor = PosProveedor::where('rut', $rutProveedorLimpio)->first();

        // Si no existe, puedes crear uno o manejar el error
        if (!$proveedor) {
            // Opcional: puedes crear el proveedor automáticamente si no existe
            $razonSocial = $this->buscarValor($cabeceras, $row, 'Razon Social');
            $proveedor = PosProveedor::create([
                'rut' => $rutProveedorLimpio,
                'nombre' => $razonSocial,
                'razon_social' => $razonSocial,
                'par_empresa_id' => $empresaId,
            ]);
        }

        $proveedorId = $proveedor->id;

        // Buscar con_sub_cuenta_id desde con_proveedor_cuentas
        $cuentaProveedor = ConProveedorCuenta::where('par_empresa_id', $empresaId)
            ->where('pos_proveedor_id', $proveedorId)
            ->first();

        $conSubCuentaId = $cuentaProveedor ? $cuentaProveedor->con_sub_cuenta_id : 1;

        // Buscar las columnas por nombre
        $registro = [
            'rut_empresa' => $rutEmpresa,
            'par_empresa_id' => $empresaId,
            'periodo' => $periodo,
            'par_periodo_id' => $periodoId,
            'pos_proveedor_id' => $proveedorId,
            'Nro' => $this->numero($this->buscarValor($cabeceras, $row, 'Nro')),
            'Tipo_Doc' => $this->buscarValor($cabeceras, $row, 'Tipo Doc'),
            'Tipo_Compra' => $this->buscarValor($cabeceras, $row, 'Tipo Compra'),
            'RUT_Proveedor' => $this->buscarValor($cabeceras, $row, 'RUT Proveedor'),
            'Razon_Social' => $this->buscarValor($cabeceras, $row, 'Razon Social'),
            'Folio' => $this->numero($this->buscarValor($cabeceras, $row, 'Folio')),
            'Fecha_Docto' => $this->formatearFecha($this->buscarValor($cabeceras, $row, 'Fecha Docto')),
            'Fecha_Recepcion' => $this->formatearFechaConHora($this->buscarValor($cabeceras, $row, 'Fecha Recepcion')),
            'Fecha_Acuse' => $this->formatearFechaConHora($this->buscarValor($cabeceras, $row, 'Fecha Acuse')),
            'Monto_Exento' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto Exento')),
            'Monto_Neto' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto Neto')),
            'Monto_IVA_Recuperable' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto IVA Recuperable')),
            'Monto_IVA_Uso_Comun' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto IVA Uso Comun')),
            'IVA_Activo_Fijo' => $this->numero($this->buscarValor($cabeceras, $row, 'IVA Activo Fijo')),
            'Monto_IVA_No_Recuperable' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto Iva No Recuperable')),
            'Codigo_IVA_No_Recuperable' => $this->buscarValor($cabeceras, $row, 'Codigo IVA No Rec.'),
            'Monto_Total' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto Total')),
            'Monto_Neto_Activo_Fijo' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto Neto Activo Fijo')),
            'IVA_uso_Comun' => $this->numero($this->buscarValor($cabeceras, $row, 'IVA uso Comun')),
            'Impto_Sin_Derecho_Credito' => $this->numero($this->buscarValor($cabeceras, $row, 'Impto Sin Derecho a Credito')),
            'IVA_No_Retenido' => $this->numero($this->buscarValor($cabeceras, $row, 'IVA No Retenido')),
            'Tabacos_Puros' => $this->numero($this->buscarValor($cabeceras, $row, 'Tabacos Puros')),
            'Tabacos_Cigarrillos' => $this->numero($this->buscarValor($cabeceras, $row, 'Tabacos Cigarrillos')),
            'Tabacos_Elaborados' => $this->numero($this->buscarValor($cabeceras, $row, 'Tabacos Elaborados')),
            'NCE_o_NDE' => $this->buscarValor($cabeceras, $row, 'NCE o NDE sobre Fact. de Compra'),
            'Codigo_Otro_Impuesto' => $this->buscarValor($cabeceras, $row, 'Codigo Otro Impuesto'),
            'Valor_Otro_Impuesto' => $this->numero($this->buscarValor($cabeceras, $row, 'Valor Otro Impuesto')),
            'Tasa_Otro_Impuesto' => $this->numero($this->buscarValor($cabeceras, $row, 'Tasa Otro Impuesto')),
        ];

        $registros[] = $registro;
    }

    fclose($file);

    // ---- Buscar duplicados ----
    $foliosExistentes = ConSiiCompra::select('RUT_Proveedor', 'Tipo_Doc', 'Folio')
        ->get()
        ->map(function ($item) {
            return "{$item->RUT_Proveedor}|{$item->Tipo_Doc}|{$item->Folio}";
        })->toArray();

    $foliosExistentes = array_flip($foliosExistentes);

    $nuevosRegistros = [];

    foreach ($registros as $registro) {
        $clave = "{$registro['RUT_Proveedor']}|{$registro['Tipo_Doc']}|{$registro['Folio']}";
        if (!isset($foliosExistentes[$clave])) {
            $nuevosRegistros[] = $registro;
        }
    }

    if (count($nuevosRegistros) > 0) {
        ConSiiCompra::insert($nuevosRegistros);
    }

    return response()->json([
        'mensaje' => 'Importación completada',
        'total_leídos' => count($registros),
        'total_insertados' => count($nuevosRegistros)
    ]);
}


    public function importarComprasDesdeCSV($nombreArchivo)
{
    Log::info("📥 Recibido archivo para importar: $nombreArchivo");

    $path = storage_path('app/public/libros/' . $nombreArchivo);

    if (!file_exists($path)) {
        Log::warning("⚠️ Archivo no encontrado en: $path");
        return response()->json(['error' => 'El archivo no existe: ' . $nombreArchivo], 404);
    }

    // Extraer RUT y periodo desde el nombre del archivo (validación estricta)
    if (preg_match('/RCV_COMPRA_REGISTRO_(.*?)_(.*?)\.csv/', $nombreArchivo, $matches) !== 1) {
        Log::warning("⚠️ Nombre de archivo inválido para importar compras: $nombreArchivo");
        return response()->json(['error' => 'Nombre de archivo inválido, formato esperado: RCV_COMPRA_REGISTRO_{RUT}_{PERIODO}.csv'], 400);
    }

    $rutEmpresaConDV = $matches[1];
    $periodo = $matches[2];

    $rutEmpresa = str_replace('-', '', $rutEmpresaConDV);
    Log::info("🏢 RUT Empresa: $rutEmpresa, Período: $periodo");

    $empresa = ParEmpresa::where('rut', $rutEmpresa)->first();

    if (!$empresa) {
        Log::warning("❌ No se encontró la empresa con el RUT: $rutEmpresa");
        return response()->json(['error' => 'No se encontró la empresa con el RUT: ' . $rutEmpresa], 404);
    }

    $empresaId = $empresa->id;

    $parPeriodo = ParPeriodo::where('mes_ano', $periodo)->first();

    if (!$parPeriodo) {
        Log::warning("❌ No se encontró el período para el valor $periodo");
        return response()->json(['error' => 'No se encontró el período para el valor ' . $periodo], 404);
    }

    $periodoId = $parPeriodo->id;

    Log::info("🔍 Empresa ID: $empresaId, TPeriodo ID: $periodoId");

    // Leer CSV
    $file = fopen($path, 'r');
    $header = fgetcsv($file, 0, ';');
    $cabeceras = array_map('trim', $header);

    $registros = [];

    while (($row = fgetcsv($file, 0, ';')) !== false) {
        if (count($row) < 5) continue;

        $rutProveedor = $this->buscarValor($cabeceras, $row, 'RUT Proveedor');
        $rutProveedorLimpio = str_replace('-', '', $rutProveedor);

        $proveedor = PosProveedor::where('rut', $rutProveedorLimpio)->first();

        if (!$proveedor) {
            $razonSocial = $this->buscarValor($cabeceras, $row, 'Razon Social');
            $proveedor = PosProveedor::create([
                'rut' => $rutProveedorLimpio,
                'nombre' => $razonSocial,
                'razon_social' => $razonSocial,
                'par_empresa_id' => $empresaId,
            ]);
            Log::info("➕ PosProveedor creado: {$proveedor->rut}");
        }

        $proveedorId = $proveedor->id;

        $cuentaProveedor = ConProveedorCuenta::where('par_empresa_id', $empresaId)
            ->where('pos_proveedor_id', $proveedorId)
            ->first();

        $conSubCuentaId = $cuentaProveedor ? $cuentaProveedor->con_sub_cuenta_id : 1;

        $registro = [
            'rut_empresa' => $rutEmpresa,
            'par_empresa_id' => $empresaId,
            'periodo' => $periodo,
            'par_periodo_id' => $periodoId,
            'pos_proveedor_id' => $proveedorId,
            'Nro' => $this->numero($this->buscarValor($cabeceras, $row, 'Nro')),
            'Tipo_Doc' => $this->buscarValor($cabeceras, $row, 'Tipo Doc'),
            'Tipo_Compra' => $this->buscarValor($cabeceras, $row, 'Tipo Compra'),
            'RUT_Proveedor' => $this->buscarValor($cabeceras, $row, 'RUT Proveedor'),
            'Razon_Social' => $this->buscarValor($cabeceras, $row, 'Razon Social'),
            'Folio' => $this->numero($this->buscarValor($cabeceras, $row, 'Folio')),
            'Fecha_Docto' => $this->formatearFecha($this->buscarValor($cabeceras, $row, 'Fecha Docto')),
            'Fecha_Recepcion' => $this->formatearFechaConHora($this->buscarValor($cabeceras, $row, 'Fecha Recepcion')),
            'Fecha_Acuse' => $this->formatearFechaConHora($this->buscarValor($cabeceras, $row, 'Fecha Acuse')),
            'Monto_Exento' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto Exento')),
            'Monto_Neto' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto Neto')),
            'Monto_IVA_Recuperable' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto IVA Recuperable')),
            'Monto_IVA_Uso_Comun' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto IVA Uso Comun')),
            'Monto_IVA_No_Recuperable' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto Iva No Recuperable')),
            'Codigo_IVA_No_Recuperable' => $this->buscarValor($cabeceras, $row, 'Codigo IVA No Rec.'),
            'Monto_Total' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto Total')),
            'Monto_Neto_Activo_Fijo' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto Neto Activo Fijo')),
            'IVA_uso_Comun' => $this->numero($this->buscarValor($cabeceras, $row, 'IVA uso Comun')),
            'Impto_Sin_Derecho_Credito' => $this->numero($this->buscarValor($cabeceras, $row, 'Impto Sin Derecho a Credito')),
            'IVA_No_Retenido' => $this->numero($this->buscarValor($cabeceras, $row, 'IVA No Retenido')),
            'Tabacos_Puros' => $this->numero($this->buscarValor($cabeceras, $row, 'Tabacos Puros')),
            'Tabacos_Cigarrillos' => $this->numero($this->buscarValor($cabeceras, $row, 'Tabacos Cigarrillos')),
            'Tabacos_Elaborados' => $this->numero($this->buscarValor($cabeceras, $row, 'Tabacos Elaborados')),
            'NCE_o_NDE' => $this->buscarValor($cabeceras, $row, 'NCE o NDE sobre Fact. de Compra'),
            'Codigo_Otro_Impuesto' => $this->buscarValor($cabeceras, $row, 'Codigo Otro Impuesto'),
            'Valor_Otro_Impuesto' => $this->numero($this->buscarValor($cabeceras, $row, 'Valor Otro Impuesto')),
            'Tasa_Otro_Impuesto' => $this->numero($this->buscarValor($cabeceras, $row, 'Tasa Otro Impuesto')),
        ];

        $registros[] = $registro;
    }

    fclose($file);

    Log::info("📊 Total registros leídos: " . count($registros));

    $foliosExistentes = ConSiiCompra::select('RUT_Proveedor', 'Tipo_Doc', 'Folio')
        ->get()
        ->map(fn($item) => "{$item->RUT_Proveedor}|{$item->Tipo_Doc}|{$item->Folio}")
        ->toArray();

    $foliosExistentes = array_flip($foliosExistentes);
    $nuevosRegistros = [];

    foreach ($registros as $registro) {
        $clave = "{$registro['RUT_Proveedor']}|{$registro['Tipo_Doc']}|{$registro['Folio']}";
        if (!isset($foliosExistentes[$clave])) {
            $nuevosRegistros[] = $registro;
        }
    }

    Log::info("✅ Total registros nuevos a insertar: " . count($nuevosRegistros));

    if (count($nuevosRegistros) > 0) {
        ConSiiCompra::insert($nuevosRegistros);
        Log::info("🟢 Se insertaron " . count($nuevosRegistros) . " registros en la base de datos.");
    }

    return response()->json([
        'mensaje' => 'Importación completada',
        'total_leídos' => count($registros),
        'total_insertados' => count($nuevosRegistros)
    ]);
}

    // Función que recibe los datos
    public function procesarDatos($rutProveedor, $rutEmpresa, $periodo)
    {
        // 1. Obtener solo la parte numérica del RUT (sin el DV)
        $rutProveedorNumerico = explode('-', $rutProveedor)[0];
        $rutEmpresaNumerico = explode('-', $rutEmpresa)[0];

        // 2. Buscar el ID del proveedor
        $proveedor = PosProveedor::where('rut', $rutProveedorNumerico)->first();
        if (!$proveedor) {
            return response()->json(['error' => 'PosProveedor no encontrado'], 404);
        }
        $proveedorId = $proveedor->id;

        // 3. Buscar el ID de la empresa
        $empresa = ParEmpresa::where('rut', $rutEmpresaNumerico)->first();
        if (!$empresa) {
            return response()->json(['error' => 'Empresa no encontrada'], 404);
        }
        $empresaId = $empresa->id;

        // 4. Buscar el con_sub_cuenta_id
        $cuentaProveedor = ConProveedorCuenta::where('pos_proveedor_id', $proveedorId)
            ->where('par_empresa_id', $empresaId)
            ->first();
        if (!$cuentaProveedor) {
            return response()->json(['error' => 'Cuenta proveedor no encontrada'], 404);
        }
        $subCuentaId = $cuentaProveedor->con_sub_cuenta_id;

        // 5. Obtener periodo_id desde par_periodos
        [$mes, $ano] = explode('-', $periodo); // ej. "05", "2024"
        $parPeriodo = ParPeriodo::where('mes', intval($mes))
            ->where('ano', intval($ano))
            ->first();
        if (!$parPeriodo) {
            return response()->json(['error' => 'Periodo no encontrado'], 404);
        }
        $periodoId = $parPeriodo->id;

        // Resultado final
        return response()->json([
            'pos_proveedor_id' => $proveedorId,
            'par_empresa_id' => $empresaId,
            'con_sub_cuenta_id' => $subCuentaId,
            'periodo_id' => $periodoId,
        ]);
    }
    private function buscarValor($cabeceras, $row, $nombreColumna)
    {
        $index = array_search($nombreColumna, $cabeceras);

        // SOLO PARA DEBUG: mostrar si encontró el índice o no
        if ($index === false) {
            Log::info("No se encontró la columna: '$nombreColumna' en el archivo CSV.");
        } else {
            Log::info("Encontrada columna '$nombreColumna' en índice: $index con valor: " . ($row[$index] ?? 'NO VALOR'));
        }

        if ($index !== false && isset($row[$index])) {
            $valor = trim($row[$index]);
            return $valor !== '' ? $valor : null;
        }
        return null;
    }

    private function formatearFecha($fecha)
    {
        if (!$fecha) return null;

        $fecha = trim($fecha); // 🔥 IMPORTANTE

        if (strpos($fecha, '/') === false) {
            return null;
        }

        [$dia, $mes, $anio] = explode('/', $fecha);

        if (!is_numeric($dia) || !is_numeric($mes) || !is_numeric($anio)) {
            return null;
        }

        if (!checkdate((int)$mes, (int)$dia, (int)$anio)) {
            return null;
        }

        return "$anio-$mes-$dia";
    }

    private function formatearFechaConHora($fecha)
    {
        if (!$fecha) return null;

        // Separar fecha y hora si existe
        $partes = explode(' ', trim($fecha));
        $soloFecha = $partes[0];
        $hora = $partes[1] ?? '00:00:00';

        if (strpos($soloFecha, '/') === false) {
            return null;
        }

        [$dia, $mes, $anio] = explode('/', $soloFecha);

        if (!is_numeric($dia) || !is_numeric($mes) || !is_numeric($anio)) {
            return null;
        }

        if (!checkdate((int)$mes, (int)$dia, (int)$anio)) {
            return null;
        }

        return sprintf('%04d-%02d-%02d %s', $anio, $mes, $dia, $hora);
    }


    private function numero($valor)
    {
        // Limpia y convierte a número. Si no es numérico, retorna 0.
        if (is_null($valor)) return 0;
        $limpio = str_replace(['$', '.', ' ', ','], '', $valor);
        return is_numeric($limpio) ? (float)$limpio : 0;
    }


public function descargar(Request $request)
{
    $request->validate([
        'par_empresa_id' => 'required|exists:par_empresas,id',
        'periodo'        => 'required|exists:par_periodos,id'
    ]);

    try {
        // ✅ Tomar token desde el header Authorization: Bearer ...
        $token = $request->bearerToken();
        if (!$token) {
            return response()->json(['error' => 'Usuario no autenticado. Proporcione un token válido.'], 401);
        }

        $empresa   = \App\Models\Parametros\ParEmpresa::findOrFail($request->par_empresa_id);
        $rutLimpio = preg_replace('/[^0-9kK]/', '', $empresa->rut);
        $rut       = substr($rutLimpio, 0, -1) . '-' . strtoupper(substr($rutLimpio, -1));
        $clave     = Crypt::decryptString($empresa->sii);

        $periodo = \App\Models\Parametros\ParPeriodo::findOrFail($request->periodo);
        [$mes, $anio] = explode('-', $periodo->nombre);

        Log::info('📥 Ejecutando script Node.js con:', [
            'rut' => $rut,
            'clave' => '[oculto]',
            'mes' => $mes,
            'anio' => $anio,
        ]);

        $comando = [
            'env',
            'NODE_PATH=/opt/node_modules/lib/node_modules',
            '/usr/bin/node',
            'scripts/descargar_sii.js',
            $rut,
            $clave,
            $mes,
            $anio,
            $token // ⬅️ ahora viene del header
        ];

        $process = new Process($comando);
        $process->setWorkingDirectory(base_path());
        $process->setTimeout(300);
        $process->run();

        if (!$process->isSuccessful()) {
            throw new ProcessFailedException($process);
        }

        $output = $process->getOutput();
        Log::info("📤 Salida del script Node.js:", [$output]);

        return response()->json([
            'message' => '✅ Script ejecutado correctamente.',
            'salida'  => $output,
        ]);
    } catch (\Throwable $e) {
        Log::error('❌ Error al ejecutar script Node.js: ' . $e->getMessage());

        return response()->json([
            'error'   => '❌ Fallo al ejecutar el script Node.js',
            'detalle' => $e->getMessage(),
        ], 500);
    }
}




    
    
}
