<?php

namespace App\Http\Controllers\Contabilidad;

use App\Http\Controllers\Controller;

use App\Models\Parametros\ParEmpresa;
use App\Models\Parametros\ParPeriodo;
use App\Models\Pos\PosProveedor;
use Illuminate\Http\Request;
use App\Models\Contabilidad\ConSiiHCompra;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Crypt;
use Symfony\Component\Process\Process;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\Process\Exception\ProcessFailedException;

class ConSiiHCompraController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return ConSiiHCompra::with(['ConSubCuenta', 'ConSubCuenta.cuenta.ConCategoriaCuenta.tipoCuenta'])->where('estado', 1)->get();
    }

    public function Filtro(Request $request)

    
    {
        $filtro = $request->filtro;
        $periodo = $request->periodo;
        return ConSiiHCompra::with(['ConSubCuenta', 'ConSubCuenta.cuenta.ConCategoriaCuenta.tipoCuenta'])->where('par_empresa_id', $filtro)->where('par_periodo_id', $periodo)->orderBy('par_periodo_id')->where('estado',1)->get();
    }

    public function importarComprasDesdeCSV($nombreArchivo)
{
    try {
        Log::info("📅 Iniciando importación de HTML compras: $nombreArchivo");

        $path = storage_path('app/public/honorarios/' . $nombreArchivo);
        if (!file_exists($path)) {
            Log::error("❌ Archivo no encontrado: $nombreArchivo");
            return response()->json(['error' => 'El archivo no existe: ' . $nombreArchivo], 404);
        }

        if (preg_match('/^HO_(.*?)_(\d{6})_(CM|CMT)\.xls$/i', $nombreArchivo, $matches) !== 1) {
            Log::error("❌ Nombre de archivo inválido para importación HTML compras: $nombreArchivo");
            return response()->json(['error' => 'Nombre de archivo inválido, formato esperado: HO_{RUT}_{YYYYMM}_{CM|CMT}.xls'], 400);
        }

        $rutEmpresaConDV = $matches[1];
        $periodo = $matches[2];

        $rutEmpresa = str_replace('-', '', $rutEmpresaConDV);
        $par_empresas = ParEmpresa::where('rut', $rutEmpresa)->first();
        if (!$par_empresas) {
            Log::error("❌ ParEmpresa no encontrada con RUT: $rutEmpresa");
            return response()->json(['error' => 'No se encontró la par_empresas con el RUT: ' . $rutEmpresa], 404);
        }

        $tPeriodo = ParPeriodo::where('mes_ano', $periodo)->first();
        if (!$tPeriodo) {
            Log::error("❌ Período no encontrado: $periodo");
            return response()->json(['error' => 'No se encontró el período: ' . $periodo], 404);
        }

        libxml_use_internal_errors(true);
        $dom = new \DOMDocument();
        $htmlContent = file_get_contents($path);
        $htmlContent = mb_convert_encoding($htmlContent, 'HTML-ENTITIES', 'UTF-8');
        $dom->loadHTML($htmlContent);

        $rows = $dom->getElementsByTagName('tr');
        $cabeceras = [];
        $registros = [];

        $totalLeidos = 0;
        $totalInsertados = 0;

        foreach ($rows as $index => $tr) {
            $tds = $tr->getElementsByTagName('td');
            if ($tds->length === 0) continue;

            $valores = [];
            foreach ($tds as $td) {
                $valores[] = trim($td->nodeValue);
            }

            // Detectar cabeceras
            if (empty($cabeceras)) {
                if (count($valores) >= 9 && in_array('Rut', $valores) && in_array('Brutos', $valores)) {
                    $cabeceras = $valores;
                    Log::info("📌 Cabeceras detectadas automáticamente: " . json_encode($cabeceras));
                    continue;
                } else {
                    continue;
                }
            }

            if (count($valores) !== count($cabeceras)) {
                Log::warning("⚠️ Fila inválida (columnas desiguales): " . json_encode($valores));
                continue;
            }

            try {
                $datos = array_combine($cabeceras, $valores);

                if (empty($datos['N°']) || empty($datos['Rut'])) {
                    Log::info("ℹ️ Fila omitida por datos vacíos: " . json_encode($datos));
                    continue;
                }

                $totalLeidos++;

                $numero = (int) $datos['N°'];
                $fecha = $this->formatearFecha($datos['Fecha']);
                $rutProveedor = str_replace('-', '', $datos['Rut']);

                $existe = \App\Models\Contabilidad\ConSiiHCompra::where('numero', $numero)
                    ->where('par_empresa_id', $par_empresas->id)
                    ->where('rut', $rutProveedor)
                    ->exists();

                if ($existe) {
                    Log::info("🔁 Registro duplicado omitido: N° $numero, RUT $rutProveedor, ParEmpresa ID {$par_empresas->id}");
                    continue;
                }

                $razonSocial = $datos['Nombre o Razón Social'] ?? '';

                $proveedor = \App\Models\Pos\PosProveedor::firstOrCreate(
                    ['rut' => $rutProveedor],
                    [
                        'nombre' => $razonSocial,
                        'razon_social' => $razonSocial,
                        'par_empresa_id' => $par_empresas->id
                    ]
                );

                $registro = [
                    'numero' => $numero,
                    'fecha' => $fecha,
                    'fecha_anulacion' => $this->formatearFecha($datos['Fecha Anulación'] ?? null),
                    'rut' => $rutProveedor,
                    'nombre_razon_social' => $razonSocial,
                    'soc_prof' => strtolower($datos['Soc. Prof.'] ?? 'no') === 'sí' || strtolower($datos['Soc. Prof.'] ?? 'no') === 'si',
                    'brutos' => (float) str_replace('.', '', $datos['Brutos']),
                    'retenido' => (float) str_replace('.', '', $datos['Retenido']),
                    'pagado' => (float) str_replace('.', '', $datos['Pagado']),
                    'par_empresa_id' => $par_empresas->id,
                    'par_periodo_id' => $tPeriodo->id,
                    'periodo' => $periodo,
                    'estado_doc' => $datos['Estado'] ?? 'VIGENTE',
                    'tipo_doc' => 'HON',
                    'pos_proveedor_id' => $proveedor->id,
                    'estado' => 1,
                ];

                $registros[] = $registro;
                $totalInsertados++;
                Log::info("✅ Registro preparado: " . json_encode($registro));
            } catch (\Throwable $e) {
                Log::error("❌ Error fila HTML compra: " . $e->getMessage() . " en fila " . ($index + 1));
            }
        }

        if (count($registros) > 0) {
            \App\Models\Contabilidad\ConSiiHCompra::insert($registros);
            Log::info("✅ Insertados en con_sii_h_compras: " . count($registros));
        } else {
            Log::warning("⚠️ No se insertó ningún registro de compras. Total registros válidos: 0");
        }

        return response()->json([
            'mensaje' => 'Importación desde HTML de compras completada.',
            'total_leídos' => $totalLeidos,
            'total_insertados' => $totalInsertados
        ]);
    } catch (\Exception $e) {
        Log::error("🔥 Error inesperado en importación de compras: " . $e->getMessage() . " en línea " . $e->getLine());
        return response()->json([
            'error' => 'Ocurrió un error durante la importación de compras',
            'detalle' => $e->getMessage(),
            'linea' => $e->getLine()
        ], 500);
    }
}





    public function importarComprasDesdeHTMLVET($nombreArchivo)
    {
        try {
            Log::info("📅 Iniciando importación de HTML compras VET: $nombreArchivo");

            $path = storage_path('app/public/honorarios/' . $nombreArchivo);
            if (!file_exists($path)) {
                Log::error("❌ Archivo no encontrado: $nombreArchivo");
                return response()->json(['error' => 'El archivo no existe: ' . $nombreArchivo], 404);
            }

            if (preg_match('/^HO_(.*?)_(\d{6})_(CMT)\.xls$/i', $nombreArchivo, $matches) !== 1) {
                Log::error("❌ Nombre de archivo inválido para importación HTML compras VET: $nombreArchivo");
                return response()->json(['error' => 'Nombre de archivo inválido, formato esperado: HO_{RUT}_{YYYYMM}_CMT.xls'], 400);
            }

            $rutEmpresaConDV = $matches[1];
            $periodo = $matches[2];

            $rutEmpresa = str_replace('-', '', $rutEmpresaConDV);
            $par_empresas = ParEmpresa::where('rut', $rutEmpresa)->first();
            if (!$par_empresas) {
                Log::error("❌ ParEmpresa no encontrada con RUT: $rutEmpresa");
                return response()->json(['error' => 'No se encontró la par_empresas con el RUT: ' . $rutEmpresa], 404);
            }

            $tPeriodo = ParPeriodo::where('mes_ano', $periodo)->first();
            if (!$tPeriodo) {
                Log::error("❌ Período no encontrado: $periodo");
                return response()->json(['error' => 'No se encontró el período: ' . $periodo], 404);
            }

            libxml_use_internal_errors(true);
            $dom = new \DOMDocument();
            $htmlContent = file_get_contents($path);
            $htmlContent = mb_convert_encoding($htmlContent, 'HTML-ENTITIES', 'UTF-8');
            $dom->loadHTML($htmlContent);

            $rows = $dom->getElementsByTagName('tr');
            $cabeceras = [];
            $registros = [];

            foreach ($rows as $index => $tr) {
                $tds = $tr->getElementsByTagName('td');
                if ($tds->length === 0) continue;

                $valores = [];
                foreach ($tds as $td) {
                    $valores[] = trim(preg_replace('/[\x00-\x1F\x7F\xA0]+/u', '', $td->nodeValue));
                }

                // Detección de cabeceras
                if (empty($cabeceras)) {
                    if (count($valores) >= 10 && in_array('Brutos', $valores) && in_array('Retenidos', $valores)) {
                        $cabeceras = array_map(function ($c) {
                            $c = preg_replace('/[\x00-\x1F\x7F\xA0]+/u', '', trim($c));
                            return match ($c) {
                                'Rut', 'Rut Receptor' => 'Rut Receptor',
                                'Nombre' => 'Nombre Receptor',
                                default => $c
                            };
                        }, $valores);

                        Log::info("📌 Cabeceras normalizadas para compras VET: " . json_encode($cabeceras));
                        continue;
                    } else {
                        continue;
                    }
                }

                // Validación de columnas
                if (count($valores) !== count($cabeceras)) {
                    if (stripos(implode('', $valores), 'total') !== false || implode('', $valores) === '') continue;

                    Log::warning("⚠️ Fila inválida (columnas desiguales): " . json_encode($valores));
                    continue;
                }

                try {
                    $datos = array_combine($cabeceras, $valores);

                    if (empty($datos['N°']) || empty($datos['Rut Receptor'])) continue;

                    $numero = (int) $datos['N°'];
                    $rutProveedor = str_replace('-', '', $datos['Rut Receptor']);
                    $fecha = $this->formatearFecha($datos['Fecha']);

                    $existe = ConSiiHCompra::where('numero', $numero)
                        ->where('par_empresa_id', $par_empresas->id)
                        ->where('rut', $rutProveedor)
                        ->exists();

                    if ($existe) {
                        Log::info("🔁 Registro duplicado omitido: par_empresa_id={$par_empresas->id}, RUT=$rutProveedor, número=$numero");
                        continue;
                    }

                    $razonSocial = $datos['Nombre Receptor'] ?? '';

                    $proveedor = PosProveedor::firstOrCreate(
                        ['rut' => $rutProveedor],
                        [
                            'nombre' => $razonSocial,
                            'razon_social' => $razonSocial,
                            'par_empresa_id' => $par_empresas->id
                        ]
                    );

                    $registro = [
                        'numero' => $numero,
                        'fecha' => $fecha,
                        'fecha_anulacion' => null,
                        'rut' => $rutProveedor,
                        'nombre_razon_social' => $razonSocial,
                        'soc_prof' => false,
                        'brutos' => (float) str_replace(['.', ','], '', $datos['Brutos']),
                        'retenido' => (float) str_replace(['.', ','], '', $datos['Retenidos']),
                        'pagado' => (float) str_replace(['.', ','], '', $datos['Pagado']),
                        'par_empresa_id' => $par_empresas->id,
                        'par_periodo_id' => $tPeriodo->id,
                        'periodo' => $periodo,
                        'estado_doc' => $datos['Estado'] ?? 'VIGENTE',
                        'tipo_doc' => 'HON',
                        'pos_proveedor_id' => $proveedor->id,
                        'estado' => 1,
                    ];

                    Log::info("✅ Registro de compra VET preparado: " . json_encode($registro));
                    $registros[] = $registro;
                } catch (\Throwable $e) {
                    Log::error("❌ Error fila HTML VET compra: " . $e->getMessage() . " en fila " . ($index + 1));
                }
            }

            if (count($registros) > 0) {
                ConSiiHCompra::insert($registros);
                Log::info("✅ Insertados en con_sii_h_compras desde VET: " . count($registros));
            } else {
                Log::warning("⚠️ No se insertó ningún registro en compras VET. Total registros válidos: 0");
            }

            return response()->json([
                'mensaje' => 'Importación desde HTML compras VET completada.',
                'total_insertados' => count($registros)
            ]);
        } catch (\Exception $e) {
            Log::error("🔥 Error inesperado en compras VET: " . $e->getMessage() . " en línea " . $e->getLine());
            return response()->json([
                'error' => 'Ocurrió un error durante la importación de compras VET',
                'detalle' => $e->getMessage(),
                'linea' => $e->getLine()
            ], 500);
        }
    }





    private function buscarValor($cabeceras, $fila, $clave)
    {
        foreach ($cabeceras as $index => $columna) {
            if (trim(mb_strtolower($columna)) === trim(mb_strtolower($clave))) {
                return $fila[$index] ?? null;
            }
        }
        return null;
    }
    private function numero($valor)
    {
        $valor = str_replace(['.', '$', ' '], '', $valor); // elimina puntos, signos y espacios
        $valor = str_replace(',', '.', $valor);            // convierte coma decimal a punto
        return is_numeric($valor) ? (float) $valor : 0;
    }

    private function formatearFecha($fecha)
    {
        if (!$fecha) return null;

        // Formato dd/mm/yyyy
        if (preg_match('/^(\d{2})\/(\d{2})\/(\d{4})$/', $fecha, $m)) {
            return "{$m[3]}-{$m[2]}-{$m[1]}";
        }

        // Formato dd-mm-yyyy
        if (preg_match('/^(\d{2})\-(\d{2})\-(\d{4})$/', $fecha, $m)) {
            return "{$m[3]}-{$m[2]}-{$m[1]}";
        }

        // Ya en formato correcto o no se puede validar
        return $fecha;
    }


    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\ConSiiHCompra  $conSiiHCompra
     * @return \Illuminate\Http\Response
     */
    public function show(ConSiiHCompra $conSiiHCompra)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\ConSiiHCompra  $conSiiHCompra
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, ConSiiHCompra $conSiiHCompra)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\ConSiiHCompra  $conSiiHCompra
     * @return \Illuminate\Http\Response
     */
    public function destroy(ConSiiHCompra $conSiiHCompra)
    {
        //
    }
  public function descargar(Request $request)
{
    $request->validate([
        'par_empresa_id' => 'required|exists:par_empresas,id',
        'periodo'    => 'required|exists:par_periodos,id'
    ]);

    try {
        // 0) Requisitos para poder ejecutar procesos
        if (!function_exists('proc_open')) {
            return response()->json([
                'error'   => 'El servidor tiene deshabilitado proc_open.',
                'detalle' => 'Habilita proc_open en php.ini (disable_functions) para usar Symfony Process.',
            ], 500);
        }

        $par_empresas = \App\Models\Parametros\ParEmpresa::findOrFail($request->par_empresa_id);
        $rutLimpio = preg_replace('/[^0-9kK]/', '', $par_empresas->rut);
        $rut = substr($rutLimpio, 0, -1) . '-' . strtoupper(substr($rutLimpio, -1));

        $clave = \Illuminate\Support\Facades\Crypt::decryptString($par_empresas->sii);

        $periodo = \App\Models\Parametros\ParPeriodo::findOrFail($request->periodo);
        [$mes, $anio] = explode('-', $periodo->nombre);

        $token = $request->bearerToken();

        $nodeBin   = '/usr/bin/node';                     // Ajusta si usas nvm u otra ruta
        $scriptAbs = base_path('scripts/honorarios.js');  // Ruta absoluta

        // (sin shell_exec) Identidad del proceso PHP
        $user = function_exists('posix_geteuid')
            ? (posix_getpwuid(posix_geteuid())['name'] ?? get_current_user())
            : get_current_user();

        Log::info('🧪 Preflight honorarios.js', [
            'node'   => $nodeBin,
            'script' => $scriptAbs,
            'exists' => file_exists($scriptAbs),
            'user'   => $user,
        ]);

        if (!file_exists($scriptAbs)) {
            return response()->json([
                'error'   => 'Script no encontrado',
                'detalle' => $scriptAbs.' no existe',
            ], 500);
        }

        // Probar Node -v (sin shell_exec)
        $probe = new \Symfony\Component\Process\Process([$nodeBin, '-v']);
        $probe->setTimeout(15);
        $probe->run();
        Log::info('🧪 Node -v', [
            'exit'   => $probe->getExitCode(),
            'stdout' => $probe->getOutput(),
            'stderr' => $probe->getErrorOutput(),
        ]);
        if (!$probe->isSuccessful()) {
            return response()->json([
                'error'  => 'Node no ejecuta',
                'stdout' => $probe->getOutput(),
                'stderr' => $probe->getErrorOutput(),
                'code'   => $probe->getExitCode(),
            ], 500);
        }

        Log::info('📥 Ejecutando script Node.js con:', [
            'rut'   => $rut,
            'clave' => '[oculto]',
            'mes'   => $mes,
            'anio'  => $anio,
            'token' => '[oculto]',
        ]);

        // ENV para el proceso hijo (ajusta si usas binario fijo de chromium)
        $env = [
            'NODE_PATH'         => '/opt/node_modules/lib/node_modules',
            'SCRIPT_TIMEOUT_MS' => '15000', // watchdog 15s en honorarios.js
            // 'PW_EXECUTABLE'   => '/home/wwwaco5/.cache/ms-playwright/chromium_headless_shell-1181/chrome-linux/headless_shell',
            // 'ETIQUETA'        => 'CM', // opcional: correr solo CM o CMT
            // 'API_BASE_URL_COMPRAS'         => config('services.api_compras'),
            // 'API_BASE_URL_COMPRAS_TOTALES' => config('services.api_compras_totales'),
        ];

        $args = [
            $nodeBin,
            $scriptAbs,
            $rut,
            $clave,
            $mes,
            $anio,
            $token,
        ];

        $p = new \Symfony\Component\Process\Process($args, base_path(), $env);
        $p->setTimeout(300);
        $p->setIdleTimeout(120);

        $t0 = microtime(true);
        $p->run();
        $dt = round((microtime(true) - $t0) * 1000);

        $stdout = $p->getOutput();
        $stderr = $p->getErrorOutput();
        $exit   = $p->getExitCode();

        Log::info("📤 STDOUT honorarios.js ({$dt} ms)", [$stdout]);
        if ($stderr) Log::warning("📄 STDERR honorarios.js ({$dt} ms)", [$stderr]);
        Log::info('🔚 Exit code honorarios.js', ['exitCode' => $exit]);

        if (!$p->isSuccessful()) {
            // Devuelve streams para depurar desde el front
            return response()->json([
                'error'  => '❌ Fallo al ejecutar honorarios.js',
                'ms'     => $dt,
                'stdout' => $stdout,
                'stderr' => $stderr,
                'code'   => $exit,
            ], 500);
        }

        return response()->json([
            'message' => '✅ Script ejecutado correctamente.',
            'ms'      => $dt,
            'stdout'  => $stdout,
            'stderr'  => $stderr, // puede venir vacío
            'code'    => $exit,
        ]);

    } catch (\Throwable $e) {
        Log::error('❌ Error descargar(): '.$e->getMessage());
        return response()->json([
            'error'   => '❌ Excepción en descargar()',
            'detalle' => $e->getMessage(),
        ], 500);
    }
}


}