<?php

namespace App\Http\Controllers\Contabilidad;
use App\Http\Controllers\Controller;

use App\Models\Parametros\ParEmpresa;
use App\Models\Parametros\ParPeriodo;
use App\Models\Pos\PosCliente;
use App\Models\Contabilidad\ConSiiHVenta;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Crypt;
use Symfony\Component\Process\Process;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Reader\Xls;
use PhpOffice\PhpSpreadsheet\Reader\Xml;
use PhpOffice\PhpSpreadsheet\Reader\Xlsx;
use Symfony\Component\Process\Exception\ProcessFailedException;

class ConSiiHVentaController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
     public function index()
    {
        return ConSiiHVenta::with(['ConSubCuenta', 'ConSubCuenta.cuenta.ConCategoriaCuenta.tipoCuenta'])->where('estado', 1)->get();
    }

    public function Filtro(Request $request)


    {
        $filtro = $request->filtro;
        $periodo = $request->periodo;
        return ConSiiHVenta::with(['ConSubCuenta', 'ConSubCuenta.cuenta.ConCategoriaCuenta.tipoCuenta'])->where('par_empresa_id', $filtro)->where('par_periodo_id', $periodo)->orderBy('par_periodo_id')->where('estado', 1)->get();
    }

   public function importarVentasDesdeCSV($nombreArchivo)
{
    try {
        Log::info("📅 Iniciando importación de HTML honorarios: $nombreArchivo");

        $path = storage_path('app/public/honorarios/' . $nombreArchivo);
        if (!file_exists($path)) {
            Log::error("❌ Archivo no encontrado: $nombreArchivo");
            return response()->json(['error' => 'El archivo no existe: ' . $nombreArchivo], 404);
        }

        if (preg_match('/^HO_(.*?)_(\d{6})_(VE|VET)\.xls$/i', $nombreArchivo, $matches) !== 1) {
            Log::error("❌ Nombre de archivo inválido para importación HTML honorarios: $nombreArchivo");
            return response()->json(['error' => 'Nombre de archivo inválido, formato esperado: HO_{RUT}_{YYYYMM}_{VE|VET}.xls'], 400);
        }

        $rutEmpresaConDV = $matches[1];
        $periodo = $matches[2];

        $rutEmpresa = str_replace('-', '', $rutEmpresaConDV);
        $empresa = ParEmpresa::where('rut', $rutEmpresa)->first();
        if (!$empresa) {
            Log::error("❌ Empresa no encontrada con RUT: $rutEmpresa");
            return response()->json(['error' => 'No se encontró la empresa con el RUT: ' . $rutEmpresa], 404);
        }

        $parPeriodo = ParPeriodo::where('mes_ano', $periodo)->first();
        if (!$parPeriodo) {
            Log::error("❌ Período no encontrado: $periodo");
            return response()->json(['error' => 'No se encontró el período: ' . $periodo], 404);
        }

        libxml_use_internal_errors(true);
        $dom = new \DOMDocument();
        $htmlContent = file_get_contents($path);
        $htmlContent = mb_convert_encoding($htmlContent, 'HTML-ENTITIES', 'UTF-8');
        $dom->loadHTML($htmlContent);

        $rows = $dom->getElementsByTagName('tr');
        $cabeceras = [];
        $registros = [];
        $nuevosRegistros = [];

        foreach ($rows as $index => $tr) {
            $tds = $tr->getElementsByTagName('td');
            if ($tds->length === 0) continue;

            $valores = [];
            foreach ($tds as $td) {
                $valores[] = trim($td->nodeValue);
            }

            if (empty($cabeceras)) {
                if (count($valores) === 13 && in_array('Rut', $valores) && in_array('Brutos', $valores)) {
                    $cabeceras = $valores;
                    Log::info("📌 Cabeceras detectadas automáticamente: " . json_encode($cabeceras));
                    continue;
                } else {
                    continue;
                }
            }

            if (count($valores) !== count($cabeceras)) {
                Log::warning("⚠️ Fila inválida (columnas desiguales): " . json_encode($valores));
                continue;
            }

            try {
                $datos = array_combine($cabeceras, $valores);

                if (empty($datos['N°']) || empty($datos['Rut'])) {
                    Log::info("ℹ️ Fila omitida por datos vacíos: " . json_encode($datos));
                    continue;
                }

                $numero = (int) $datos['N°'];
                $fecha = $this->formatearFecha($datos['Fecha Boleta']);
                $rutCliente = str_replace('-', '', $datos['Rut']);

                $existe = ConSiiHVenta::where('numero', $numero)
                    ->where('fecha', $fecha)
                    ->where('rut', $rutCliente)
                    ->exists();

                if ($existe) {
                    Log::info("🔁 Registro duplicado omitido: N° $numero, RUT $rutCliente, Fecha $fecha");
                    continue;
                }

                $razonSocial = $datos['Nombre o Razón Social'] ?? '';

                $cliente = PosCliente::firstOrCreate(
                    ['rut' => $rutCliente],
                    [
                        'nombre' => $razonSocial,
                        'razon_social' => $razonSocial,
                        'par_empresa_id' => $empresa->id
                    ]
                );

                $registro = [
                    'numero' => $numero,
                    'fecha' => $fecha,
                    'fecha_anulacion' => $this->formatearFecha($datos['Fecha Anulación']),
                    'rut' => $rutCliente,
                    'nombre_razon_social' => $razonSocial,
                    'soc_prof' => strtolower($datos['Soc. Prof.']) === 'sí' || strtolower($datos['Soc. Prof.']) === 'si',
                    'brutos' => (int) str_replace('.', '', $datos['Brutos']),
                    'retenido' => (int) str_replace('.', '', $datos['Retención emisor']),
                    'retenido_receptor' => (int) str_replace('.', '', $datos['Retención receptor']),
                    'pagado' => (int) str_replace('.', '', $datos['Líquidos']),
                    'par_empresa_id' => $empresa->id,
                    'par_periodo_id' => $parPeriodo->id,
                    'periodo' => $periodo,
                    'estado_doc' => $datos['Estado'] ?? 'VIGENTE',
                    'tipo_doc' => 'HON',
                    'pos_cliente_id' => $cliente->id,
                ];

                $registros[] = $registro;
                $nuevosRegistros[] = $registro;
            } catch (\Throwable $e) {
                Log::error("❌ Error fila HTML: " . $e->getMessage() . " en fila " . ($index + 1));
            }
        }

        if (count($nuevosRegistros) > 0) {
            ConSiiHVenta::insert($nuevosRegistros);
            Log::info("✅ Insertados en con_sii_h_ventas: " . count($nuevosRegistros));
        } else {
            Log::warning("⚠️ No se insertó ningún registro nuevo. Total registros válidos: 0");
        }

        return response()->json([
            'mensaje' => 'Importación desde HTML completada.',
            'total_leídos' => count($registros),
            'total_insertados' => count($nuevosRegistros)
        ]);
    } catch (\Exception $e) {
        Log::error("🔥 Error inesperado: " . $e->getMessage() . " en línea " . $e->getLine());
        return response()->json([
            'error' => 'Ocurrió un error durante la importación',
            'detalle' => $e->getMessage(),
            'linea' => $e->getLine()
        ], 500);
    }
}


    public function importarVentasDesdeHTMLVET($nombreArchivo)
    {
        try {
            Log::info("📅 Iniciando importación de HTML honorarios VET: $nombreArchivo");

            $path = storage_path('app/public/honorarios/' . $nombreArchivo);
            if (!file_exists($path)) {
                Log::error("❌ Archivo no encontrado: $nombreArchivo");
                return response()->json(['error' => 'El archivo no existe: ' . $nombreArchivo], 404);
            }

            if (preg_match('/^HO_(.*?)_(\d{6})_(VET)\.xls$/i', $nombreArchivo, $matches) !== 1) {
                Log::error("❌ Nombre de archivo inválido para importación HTML honorarios VET: $nombreArchivo");
                return response()->json(['error' => 'Nombre de archivo inválido, formato esperado: HO_{RUT}_{YYYYMM}_VET.xls'], 400);
            }

            $rutEmpresaConDV = $matches[1];
            $periodo = $matches[2];

            $rutEmpresa = str_replace('-', '', $rutEmpresaConDV);
            $empresa = ParEmpresa::where('rut', $rutEmpresa)->first();
            if (!$empresa) {
                Log::error("❌ Empresa no encontrada con RUT: $rutEmpresa");
                return response()->json(['error' => 'No se encontró la empresa con el RUT: ' . $rutEmpresa], 404);
            }

            $parPeriodo = ParPeriodo::where('mes_ano', $periodo)->first();
            if (!$parPeriodo) {
                Log::error("❌ Período no encontrado: $periodo");
                return response()->json(['error' => 'No se encontró el período: ' . $periodo], 404);
            }

            libxml_use_internal_errors(true);
            $dom = new \DOMDocument();
            $htmlContent = file_get_contents($path);
            $htmlContent = mb_convert_encoding($htmlContent, 'HTML-ENTITIES', 'UTF-8');
            $dom->loadHTML($htmlContent);

            $rows = $dom->getElementsByTagName('tr');
            $cabeceras = [];
            $registros = [];

            foreach ($rows as $index => $tr) {
                $tds = $tr->getElementsByTagName('td');
                if ($tds->length === 0) continue;

                $valores = [];
                foreach ($tds as $td) {
                    $valores[] = trim($td->nodeValue);
                }

                // Detectar cabecera automáticamente para VET
                if (empty($cabeceras)) {
                    if (count($valores) === 9 && in_array('Brutos', $valores) && in_array('Retenidos', $valores)) {
                        $cabeceras = $valores;
                        Log::info("📌 Cabeceras detectadas automáticamente para VET: " . json_encode($cabeceras));
                        continue;
                    } else {
                        continue;
                    }
                }

                if (count($valores) !== count($cabeceras)) {
                    Log::warning("⚠️ Fila inválida (columnas desiguales): " . json_encode($valores));
                    continue;
                }

                try {
                    $datos = array_combine($cabeceras, $valores);

                    if (empty($datos['N°']) || empty($datos['Rut'])) {
                        Log::info("ℹ️ Fila omitida por datos vacíos: " . json_encode($datos));
                        continue;
                    }

                    $numero = (int) $datos['N°'];
                    $rutCliente = str_replace('-', '', $datos['Rut']);

                    // Verificar si ya existe el registro por par_periodo_id + rut + numero
                    $existe = ConSiiHVenta::where('par_empresa_id', $empresa->id)
                        ->where('rut', $rutCliente)
                        ->where('numero', $numero)
                        ->exists();

                    if ($existe) {
                        Log::info("🔁 Registro duplicado omitido: par_empresa_id={$empresa->id}, RUT=$rutCliente, número=$numero");
                        continue;
                    }

                    $razonSocial = $datos['Nombre'] ?? '';

                    $cliente = PosCliente::firstOrCreate(
                        ['rut' => $rutCliente],
                        [
                            'nombre' => $razonSocial,
                            'razon_social' => $razonSocial,
                            'par_empresa_id' => $empresa->id
                        ]
                    );

                    $registro = [
                        'numero' => $numero,
                        'fecha' => $this->formatearFecha($datos['Fecha']),
                        'fecha_anulacion' => null,
                        'rut' => $rutCliente,
                        'nombre_razon_social' => $razonSocial,
                        'soc_prof' => false,
                        'brutos' => (int) str_replace(['.', ','], '', $datos['Brutos']),
                        'retenido' => (int) str_replace(['.', ','], '', $datos['Retenidos']),
                        'pagado' => (int) str_replace(['.', ','], '', $datos['Pagado']),
                        'par_empresa_id' => $empresa->id,
                        'par_periodo_id' => $parPeriodo->id,
                        'periodo' => $periodo,
                        'estado_doc' => $datos['Estado'] ?? 'VIGENTE',
                        'tipo_doc' => 'HON',
                        'pos_cliente_id' => $cliente->id,
                    ];

                    Log::info("✅ Registro VET preparado: " . json_encode($registro));
                    $registros[] = $registro;
                } catch (\Throwable $e) {
                    Log::error("❌ Error fila HTML VET: " . $e->getMessage() . " en fila " . ($index + 1));
                }
            }

            if (count($registros) > 0) {
                ConSiiHVenta::insert($registros);
                Log::info("✅ Insertados en con_sii_h_ventas desde VET: " . count($registros));
            } else {
                Log::warning("⚠️ No se insertó ningún registro en VET. Total registros válidos: 0");
            }

            return response()->json([
                'mensaje' => 'Importación desde HTML VET completada.',
                'total_insertados' => count($registros)
            ]);
        } catch (\Exception $e) {
            Log::error("🔥 Error inesperado en VET: " . $e->getMessage() . " en línea " . $e->getLine());
            return response()->json([
                'error' => 'Ocurrió un error durante la importación VET',
                'detalle' => $e->getMessage(),
                'linea' => $e->getLine()
            ], 500);
        }
    }

    private function buscarValor($cabeceras, $fila, $clave)
    {
        foreach ($cabeceras as $index => $columna) {
            if (trim(mb_strtolower($columna)) === trim(mb_strtolower($clave))) {
                return $fila[$index] ?? null;
            }
        }
        return null;
    }
    private function numero($valor)
    {
        $valor = str_replace(['.', '$', ' '], '', $valor); // elimina puntos, signos y espacios
        $valor = str_replace(',', '.', $valor);            // convierte coma decimal a punto
        return is_numeric($valor) ? (float) $valor : 0;
    }

    private function formatearFecha($fecha)
    {
        if (!$fecha) return null;

        // Formato dd/mm/yyyy
        if (preg_match('/^(\d{2})\/(\d{2})\/(\d{4})$/', $fecha, $m)) {
            return "{$m[3]}-{$m[2]}-{$m[1]}";
        }

        // Formato dd-mm-yyyy
        if (preg_match('/^(\d{2})\-(\d{2})\-(\d{4})$/', $fecha, $m)) {
            return "{$m[3]}-{$m[2]}-{$m[1]}";
        }

        // Ya en formato correcto o no se puede validar
        return $fecha;
    }




    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\ConSiiHVenta  $conSiiHVenta
     * @return \Illuminate\Http\Response
     */
    public function show(ConSiiHVenta $conSiiHVenta)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\ConSiiHVenta  $conSiiHVenta
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, ConSiiHVenta $conSiiHVenta)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\ConSiiHVenta  $conSiiHVenta
     * @return \Illuminate\Http\Response
     */
    public function destroy(ConSiiHVenta $conSiiHVenta)
    {
        //
    }
  public function descargar(Request $request)
{
    $request->validate([
        'par_empresa_id' => 'required|exists:par_empresas,id',
        'periodo' => 'required|exists:par_periodos,id'
    ]);

    try {
        // Obtener empresa y rut con guión
        $empresa = \App\Models\Parametros\ParEmpresa::findOrFail($request->par_empresa_id);
        $rutLimpio = preg_replace('/[^0-9kK]/', '', $empresa->rut);
        $rut = substr($rutLimpio, 0, -1) . '-' . strtoupper(substr($rutLimpio, -1));

        // 🔓 Desencriptar clave SII
        $clave = Crypt::decryptString($empresa->sii);

        // Obtener nombre del periodo y separar en mes y año
        $periodo = \App\Models\Parametros\ParPeriodo::findOrFail($request->periodo); // Ej: "Mayo-2025"
        [$mes, $anio] = explode('-', $periodo->nombre);

        // 🔐 Obtener el token del header Authorization
        $token = $request->bearerToken();

        Log::info('📥 Ejecutando script Node.js con:', [
            'rut' => $rut,
            'clave' => '[oculto]',
            'mes' => $mes,
            'anio' => $anio,
            'token' => $token
        ]);

        $comando = [
            'env',
            'NODE_PATH=/opt/node_modules/lib/node_modules',
            '/usr/bin/node',
            'scripts/honorariosv.js',
            $rut,
            $clave,
            $mes,
            $anio,
            $token // ✅ se pasa el token como argumento al script
        ];

        $process = new Process($comando);
        $process->setWorkingDirectory(base_path());
        $process->setTimeout(300); // 5 minutos

        $process->run();

        if (!$process->isSuccessful()) {
            throw new ProcessFailedException($process);
        }

        $output = $process->getOutput();
        Log::info("📤 Salida del script Node.js:", [$output]);

        return response()->json([
            'message' => '✅ Script ejecutado correctamente.',
            'salida' => $output,
        ]);
    } catch (\Throwable $e) {
        Log::error('❌ Error al ejecutar script Node.js: ' . $e->getMessage());

        return response()->json([
            'error' => '❌ Fallo al ejecutar el script Node.js',
            'detalle' => $e->getMessage(),
        ], 500);
    }
}

}