<?php

namespace App\Http\Controllers\Contabilidad;

use App\Models\Proveedor;
use Illuminate\Http\Request;
use App\Models\Pos\PosProveedor;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Models\Parametros\ParPeriodo;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Schema;
use Symfony\Component\Process\Process;
use App\Models\Contabilidad\ConSiiPendiente;
use App\Models\Parametros\ParEmpresa as ParEmpresa;
use Symfony\Component\Process\Exception\ProcessFailedException;

class ConSiiPendienteController extends Controller
{
    public function index()
    {
        return ConSiiPendiente::with('Proveedor')->where('estado', 1)->get();
    }

    public function Filtro(Request $request)
    {
        $filtro = $request->input('filtro');
        $periodo = $request->input('periodo');

        $query = ConSiiPendiente::with('Proveedor')
            ->where('par_empresa_id', $filtro)
            ->where('estado', 1);

        $col = Schema::hasColumn('con_sii_pendientes', 'par_periodo_id') ? 'par_periodo_id' : 'par_periodo_id';
        $query->where($col, $periodo)->orderBy($col);

        $pendientes = $query->get();

        $facturas = $pendientes->where('tipo_doc', 33);
        $notasCredito = $pendientes->where('tipo_doc', 61);

        return response()->json([
            'par_empresa_id' => $filtro,
            'periodo' => $periodo,
            'total' => $pendientes->count(),
            'data' => $pendientes,
            'resumen' => [
                'cantidad_facturas' => $facturas->count(),
                'monto_facturas' => $facturas->sum('monto_total'),
                'cantidad_notas' => $notasCredito->count(),
                'monto_notas' => $notasCredito->sum('monto_total'),
            ]
        ]);
    }

    public function importarPendientesDesdeCSV($nombreArchivo)
    {
        $path = storage_path('app/public/libros/' . $nombreArchivo);
        if (!file_exists($path)) {
            return response()->json(['error' => 'El archivo no existe: ' . $nombreArchivo], 404);
        }

        if (preg_match('/RCV_COMPRA_PENDIENTE_(.*?)_(.*?)\.csv/', $nombreArchivo, $matches) !== 1) {
            Log::warning("⚠️ Nombre de archivo inválido para importación pendientes: $nombreArchivo");
            return response()->json(['error' => 'Nombre de archivo inválido, formato esperado: RCV_COMPRA_PENDIENTE_{RUT}_{PERIODO}.csv'], 400);
        }

        $rutEmpresaConDV = $matches[1];
        $periodo = $matches[2];

        $rutEmpresa = str_replace('-', '', $rutEmpresaConDV);
        $empresa = ParEmpresa::where('rut', $rutEmpresa)->first();
        if (!$empresa) return response()->json(['error' => 'Empresa no encontrada'], 404);

        $parPeriodo = ParPeriodo::where('mes_ano', $periodo)->first();
        if (!$parPeriodo) return response()->json(['error' => 'Período no encontrado'], 404);

        $empresaId = $empresa->id;
        $periodoId = $parPeriodo->id;

        $col = Schema::hasColumn('con_sii_pendientes', 'par_periodo_id') ? 'par_periodo_id' : 'par_periodo_id';
        ConSiiPendiente::where('par_empresa_id', $empresaId)->where($col, $periodoId)->delete();

        $file = fopen($path, 'r');
        $header = fgetcsv($file, 0, ';');
        $cabeceras = $header ? array_map('trim', $header) : [];

        $registros = [];
        while (($row = fgetcsv($file, 0, ';')) !== false) {
            if (count($row) < 1) continue;

            $rutProveedor = $this->buscarValor($cabeceras, $row, 'RUT Proveedor');
            $rutProveedorLimpio = $rutProveedor ? str_replace('-', '', $rutProveedor) : null;

            $proveedor = null;
            if ($rutProveedorLimpio) {
                $proveedor = PosProveedor::firstOrCreate(
                    ['rut' => $rutProveedorLimpio],
                    ['nombre' => $this->buscarValor($cabeceras, $row, 'Razon Social'), 'par_empresa_id' => $empresaId]
                );
            }

            $registro = [
                'rut_proveedor' => $rutProveedorLimpio,
                'razon_social' => $this->buscarValor($cabeceras, $row, 'Razon Social'),
                'proveedor_id' => $proveedor ? $proveedor->id : null,
                'folio' => $this->numero($this->buscarValor($cabeceras, $row, 'Folio')),
                'fecha_docto' => $this->formatearFecha($this->buscarValor($cabeceras, $row, 'Fecha Docto')),
                'monto_total' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto Total')),
                'par_empresa_id' => $empresaId,
                'estado' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ];

            $registro[$col] = $periodoId;
            $registros[] = $registro;
        }
        fclose($file);

        $foliosExistentes = ConSiiPendiente::pluck('folio')->toArray();
        $nuevosRegistros = array_filter($registros, function ($r) use ($foliosExistentes) {
            return empty($r['folio']) ? false : !in_array($r['folio'], $foliosExistentes);
        });

        if (!empty($nuevosRegistros)) ConSiiPendiente::insert($nuevosRegistros);

        return response()->json(['mensaje' => 'Importación completada', 'total_leidos' => count($registros), 'total_insertados' => count($nuevosRegistros)]);
    }

    private function buscarValor($cabeceras, $row, $clave)
    {
        $index = array_search($clave, $cabeceras);
        return $index !== false && isset($row[$index]) ? trim($row[$index]) : null;
    }

    private function numero($valor)
    {
        if ($valor === null) return 0;
        $v = str_replace(['.', ','], ['', '.'], $valor);
        return is_numeric($v) ? floatval($v) : 0;
    }

    private function formatearFecha($fecha)
    {
        if (!$fecha) return null;
        $d = str_replace('/', '-', $fecha);
        return date('Y-m-d', strtotime($d));
    }

    public function store(Request $request) { }
    public function show(ConSiiPendiente $conSiiPendiente) { }
    public function update(Request $request, ConSiiPendiente $conSiiPendiente) { }
    public function destroy(ConSiiPendiente $conSiiPendiente) { }

   public function descargar(Request $request)
    {
        $request->validate([
            'par_empresa_id' => 'required|exists:par_empresas,id',
            'periodo' => 'required|exists:par_periodos,id'
        ]);

        try {
            // Obtener empresa y rut con guión
            $empresa = \App\Models\Parametros\ParEmpresa::findOrFail($request->par_empresa_id);
            $rutLimpio = preg_replace('/[^0-9kK]/', '', $empresa->rut);
            $rut = substr($rutLimpio, 0, -1) . '-' . strtoupper(substr($rutLimpio, -1));

            // 🔓 Desencriptar clave SII
            $clave = Crypt::decryptString($empresa->sii);

            // Obtener nombre del periodo y separar en mes y año
            $periodo = \App\Models\Parametros\ParPeriodo::findOrFail($request->periodo); // Ej: "Mayo-2025"
            [$mes, $anio] = explode('-', $periodo->nombre);

            // 🔐 Obtener token desde encabezado Authorization
            $token = str_replace('Bearer ', '', $request->header('Authorization'));

            Log::info('📥 Ejecutando script Node.js con:', [
                'rut' => $rut,
                'clave' => '[oculto]',
                'mes' => $mes,
                'anio' => $anio,
                'par_empresa_id' => $request->par_empresa_id,
                'periodo' => $request->periodo,
                'token' => $token,
            ]);

            // Ejecutar script con 7 parámetros esperados
            $comando = [
                'env',
                'NODE_PATH=/opt/node_modules/lib/node_modules',
                '/usr/bin/node',
                'scripts/descargar_pendiente_sii.js',
                $rut,
                $clave,
                $mes,
                $anio,
                $request->par_empresa_id,
                $request->periodo,
                $token // 🔐 TOKEN AL FINAL
            ];

            $process = new Process($comando);
            $process->setWorkingDirectory(base_path());
            $process->setTimeout(300); // 5 minutos
            $process->run();

            if (!$process->isSuccessful()) {
                throw new ProcessFailedException($process);
            }

            $output = $process->getOutput();
            Log::info("📤 Salida del script Node.js:", [$output]);

            return response()->json([
                'message' => '✅ Script ejecutado correctamente.',
                'salida' => $output,
            ]);
        } catch (\Throwable $e) {
            Log::error('❌ Error al ejecutar script Node.js: ' . $e->getMessage());

            return response()->json([
                'error' => '❌ Fallo al ejecutar el script Node.js',
                'detalle' => $e->getMessage(),
            ], 500);
        }
    }

    public function aceptar(Request $request)
    {
        $request->validate(['par_empresa_id' => 'required|exists:par_empresas,id', 'periodo' => 'required|integer', 'facturas' => 'required|array']);
        try {
            $empresa = ParEmpresa::findOrFail($request->par_empresa_id);
            $clave = Crypt::decryptString($empresa->sii);
            $periodo = ParPeriodo::findOrFail($request->periodo);
            [$mes, $anio] = explode('-', $periodo->mes_ano ?? $periodo->nombre ?? '01-1970');
            $facturasJson = json_encode($request->facturas);
            $token = str_replace('Bearer ', '', $request->header('Authorization'));
            $comando = ['env', 'NODE_PATH=/opt/node_modules/lib/node_modules', '/usr/bin/node', 'scripts/aceptar_facturas_sii.js', $empresa->rut, $clave, $mes, $anio, $request->par_empresa_id, $request->periodo, $facturasJson, $token];
            $process = new Process($comando);
            $process->setWorkingDirectory(base_path());
            $process->setTimeout(300);
            $process->run();
            if (!$process->isSuccessful()) throw new ProcessFailedException($process);
            $output = $process->getOutput();
            return response()->json(['message' => 'Script ejecutado correctamente.', 'salida' => $output]);
        } catch (\Throwable $e) {
            Log::error('Error al ejecutar script Node.js: ' . $e->getMessage());
            return response()->json(['error' => 'Fallo al ejecutar el script Node.js', 'detalle' => $e->getMessage()], 500);
        }
    }

    public function actualizarPendientes(Request $request)
    {
        $request->validate(['par_empresa_id' => 'required|integer', 'par_periodo_id' => 'required|integer']);
        $empresaId = $request->input('par_empresa_id');
        $periodoId = $request->input('par_periodo_id');
        $col = Schema::hasColumn('con_sii_pendientes', 'par_periodo_id') ? 'par_periodo_id' : 'par_periodo_id';
        $registros = DB::table('con_sii_pendientes')->where('par_empresa_id', $empresaId)->where($col, $periodoId)->get();
        return response()->json(['count' => count($registros)]);
    }
}
