<?php

namespace App\Http\Controllers\Contabilidad;

use Carbon\Carbon;
use PSpell\Config;
use App\Http\Controllers\Controller;
use App\Models\Parametros\ParEmpresa;
use App\Models\Parametros\ParPeriodo;
use App\Models\Pos\PosCliente;
use App\Models\Contabilidad\ConSiiVenta;
use Illuminate\Http\Request;
use App\Models\Contabilidad\ConClienteCuenta;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Crypt;
use Symfony\Component\Process\Process;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class ConSiiVentaController extends Controller
{
    public function index()
    {
        return ConSiiVenta::with(['ConSubCuenta', 'ConSubCuenta.cuenta.ConCategoriaCuenta.tipoCuenta'])->where('estado', 1)->get();
    }

    public function Filtro(Request $request)


    {
        $filtro = $request->filtro;
        $periodo = $request->periodo;
        return ConSiiVenta::with(['ConSubCuenta', 'ConSubCuenta.cuenta.ConCategoriaCuenta.tipoCuenta'])->where('par_empresa_id', $filtro)->where('par_periodo_id', $periodo)->orderBy('par_periodo_id')->where('estado', 1)->get();
    }

      public function importarVentasDesdeCSV($nombreArchivo)
{
    try {
        Log::info("📥 Iniciando importación de ventas: $nombreArchivo");

        $path = storage_path('app/public/libros/' . $nombreArchivo);
        if (!file_exists($path)) {
            Log::warning("⚠️ Archivo no encontrado: $path");
            return response()->json(['error' => 'El archivo no existe: ' . $nombreArchivo], 404);
        }

        // Extraer rut y periodo desde el nombre del archivo de forma segura
        if (preg_match('/RCV_VENTA_(.*?)_(.*?)\.csv/i', $nombreArchivo, $matches) !== 1) {
            Log::error("❌ Nombre de archivo inválido para importar ventas: $nombreArchivo");
            return response()->json(['error' => 'Nombre de archivo inválido, formato esperado: RCV_VENTA_{RUT}_{PERIODO}.csv'], 400);
        }

        $rutEmpresaConDV = $matches[1] ?? null;
        $periodo = $matches[2] ?? null;
        Log::info("🔍 Extraído RUT DV: $rutEmpresaConDV | Periodo: $periodo");

        $rutEmpresa = str_replace('-', '', $rutEmpresaConDV);
        $empresa = ParEmpresa::where('rut', $rutEmpresa)->first();
        if (!$empresa) {
            Log::error("❌ ParEmpresa no encontrada con RUT: $rutEmpresa");
            return response()->json(['error' => 'No se encontró la empresa con el RUT: ' . $rutEmpresa], 404);
        }

        $empresaId = $empresa->id;
        Log::info("🏢 ParEmpresa ID: $empresaId");

        $tPeriodo = ParPeriodo::where('mes_ano', $periodo)->first();
        if (!$tPeriodo) {
            Log::error("❌ Periodo no encontrado: $periodo");
            return response()->json(['error' => 'No se encontró el período para el valor ' . $periodo], 404);
        }

        $periodoId = $tPeriodo->id;
        Log::info("📅 Periodo ID: $periodoId");

        $file = fopen($path, 'r');
        $cabeceras = fgetcsv($file, 0, ';');
        $registros = [];
        $contador = 0;

        while (($row = fgetcsv($file, 0, ';')) !== false) {
            if (count($row) < 5) continue;

            $contador++;
            try {
                $rutCliente = $this->buscarValor($cabeceras, $row, 'Rut cliente');
                $rutClienteLimpio = str_replace('-', '', $rutCliente);

                $cliente = PosCliente::where('rut', $rutClienteLimpio)->first();
                if (!$cliente) {
                    $razonSocial = $this->buscarValor($cabeceras, $row, 'Razon Social');
                    $cliente = PosCliente::create([
                        'rut' => $rutClienteLimpio,
                        'nombre' => $razonSocial,
                        'razon_social' => $razonSocial,
                        'par_empresa_id' => $empresaId,
                    ]);
                    Log::info("🆕 Cliente creado: $rutClienteLimpio ($razonSocial)");
                }

                $clienteId = $cliente->id;
                // Buscar la cuenta del cliente; la columna correcta es pos_cliente_id
                $cuentaCliente = ConClienteCuenta::where('par_empresa_id', $empresaId)
                    ->where('pos_cliente_id', $clienteId)
                    ->first();

                $conSubCuentaId = $cuentaCliente ? $cuentaCliente->con_sub_cuenta_id : 1;

                $registro = [
                    'rut_empresa' => $rutEmpresa,
                    'par_empresa_id' => $empresaId,
                    'periodo' => $periodo,
                    'par_periodo_id' => $periodoId,
                    'pos_cliente_id' => $clienteId,
                    'Nro' => $this->numero($this->buscarValor($cabeceras, $row, 'Nro')),
                    'Tipo_Doc' => $this->buscarValor($cabeceras, $row, 'Tipo Doc'),
                    'Tipo_Venta' => $this->buscarValor($cabeceras, $row, 'Tipo Venta'),
                    'Rut_cliente' => $this->buscarValor($cabeceras, $row, 'Rut cliente'),
                    'Razon_Social' => $this->buscarValor($cabeceras, $row, 'Razon Social'),
                    'Folio' => $this->buscarValor($cabeceras, $row, 'Folio'),
                    'Fecha_Docto' => $this->formatearFecha($this->buscarValor($cabeceras, $row, 'Fecha Docto')),
                    'Fecha_Recepcion' => $this->formatearFecha($this->buscarValor($cabeceras, $row, 'Fecha Recepcion')),
                    'Fecha_Acuse_Recibo' => $this->formatearFecha($this->buscarValor($cabeceras, $row, 'Fecha Acuse Recibo')),
                    'Fecha_Reclamo' => $this->formatearFecha($this->buscarValor($cabeceras, $row, 'Fecha Reclamo')),
                    'Monto_Exento' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto Exento')),
                    'Monto_Neto' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto Neto')),
                    'Monto_IVA' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto IVA')),
                    'Monto_total' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto total')),
                    'IVA_Retenido_Total' => $this->numero($this->buscarValor($cabeceras, $row, 'IVA Retenido Total')),
                    'IVA_Retenido_Parcial' => $this->numero($this->buscarValor($cabeceras, $row, 'IVA Retenido Parcial')),
                    'IVA_no_retenido' => $this->numero($this->buscarValor($cabeceras, $row, 'IVA no retenido')),
                    'IVA_propio' => $this->numero($this->buscarValor($cabeceras, $row, 'IVA propio')),
                    'IVA_Terceros' => $this->numero($this->buscarValor($cabeceras, $row, 'IVA Terceros')),
                    'RUT_Emisor_Liquid_Factura' => $this->buscarValor($cabeceras, $row, 'RUT Emisor Liquid. Factura'),
                    'Neto_Comision_Liquid_Factura' => $this->numero($this->buscarValor($cabeceras, $row, 'Neto Comision Liquid. Factura')),
                    'Exento_Comision_Liquid_Factura' => $this->numero($this->buscarValor($cabeceras, $row, 'Exento Comision Liquid. Factura')),
                    'IVA_Comision_Liquid_Factura' => $this->numero($this->buscarValor($cabeceras, $row, 'IVA Comision Liquid. Factura')),
                    'IVA_fuera_de_plazo' => $this->numero($this->buscarValor($cabeceras, $row, 'IVA fuera de plazo')),
                    'Tipo_Docto_Referencia' => $this->buscarValor($cabeceras, $row, 'Tipo Docto. Referencia'),
                    'Folio_Docto_Referencia' => $this->buscarValor($cabeceras, $row, 'Folio Docto. Referencia'),
                    'Num_Ident_Receptor_Extranjero' => $this->buscarValor($cabeceras, $row, 'Num. Ident. Receptor Extranjero'),
                    'Nacionalidad_Receptor_Extranjero' => $this->buscarValor($cabeceras, $row, 'Nacionalidad Receptor Extranjero'),
                    'Credito_empresa_constructora' => $this->numero($this->buscarValor($cabeceras, $row, 'Credito empresa constructora')),
                    'Impto_Zona_Franca' => $this->numero($this->buscarValor($cabeceras, $row, 'Impto. Zona Franca (Ley 18211)')),
                    'Garantia_Dep_Envases' => $this->numero($this->buscarValor($cabeceras, $row, 'Garantia Dep. Envases')),
                    'Indicador_Venta_sin_Costo' => $this->buscarValor($cabeceras, $row, 'Indicador Venta sin Costo'),
                    'Indicador_Servicio_Periodico' => $this->buscarValor($cabeceras, $row, 'Indicador Servicio Periodico'),
                    'Monto_No_facturable' => $this->numero($this->buscarValor($cabeceras, $row, 'Monto No facturable')),
                    'Total_Monto_Periodo' => $this->numero($this->buscarValor($cabeceras, $row, 'Total Monto Periodo')),
                    'Venta_Pasajes_Transporte_Nacional' => $this->numero($this->buscarValor($cabeceras, $row, 'Venta Pasajes Transporte Nacional')),
                    'Venta_Pasajes_Transporte_Internacional' => $this->numero($this->buscarValor($cabeceras, $row, 'Venta Pasajes Transporte Internacional')),
                    'Numero_Interno' => $this->buscarValor($cabeceras, $row, 'Numero Interno'),
                    'Codigo_Sucursal' => $this->buscarValor($cabeceras, $row, 'Codigo Sucursal'),
                    'NCE_o_NDE_sobre_Fact_de_Compra' => $this->buscarValor($cabeceras, $row, 'NCE o NDE sobre Fact. de Compra'),
                    'Codigo_Otro_Imp' => $this->buscarValor($cabeceras, $row, 'Codigo Otro Imp.'),
                    'Valor_Otro_Imp' => $this->numero($this->buscarValor($cabeceras, $row, 'Valor Otro Imp.')),
                    'Tasa_Otro_Imp' => $this->numero($this->buscarValor($cabeceras, $row, 'Tasa Otro Imp.')),
                ];

                $registros[] = $registro;
            } catch (\Exception $exRow) {
                Log::error("❌ Error al procesar fila #$contador: " . $exRow->getMessage());
            }
        }

        fclose($file);
        Log::info("📊 Total filas procesadas: $contador");

        $foliosExistentes = ConSiiVenta::select('Rut_cliente', 'Tipo_Doc', 'Folio')
            ->get()
            ->map(fn($item) => "{$item->Rut_cliente}|{$item->Tipo_Doc}|{$item->Folio}")
            ->toArray();
        $foliosExistentes = array_flip($foliosExistentes);

        $nuevosRegistros = array_filter($registros, function ($registro) use ($foliosExistentes) {
            $clave = "{$registro['Rut_cliente']}|{$registro['Tipo_Doc']}|{$registro['Folio']}";
            return !isset($foliosExistentes[$clave]);
        });

        Log::info("🆕 Registros nuevos: " . count($nuevosRegistros));

        if (count($nuevosRegistros) > 0) {
            ConSiiVenta::insert($nuevosRegistros);
            Log::info("✅ Insertados en base de datos: " . count($nuevosRegistros));
        }

        return response()->json([
            'mensaje' => 'Importación completada',
            'total_leídos' => count($registros),
            'total_insertados' => count($nuevosRegistros)
        ]);
    } catch (\Exception $e) {
        Log::error("❌ Error general al importar ventas: " . $e->getMessage());
        return response()->json([
            'error' => 'Ocurrió un error al importar',
            'detalle' => $e->getMessage(),
            'linea' => $e->getLine()
        ], 500);
    }
}
  

public function importarResumenBoletaDesdeCSV($nombreArchivo)
{
    try {
        Log::info("📅 [RESUMEN-BOLETAS] Iniciando importación: {$nombreArchivo}");

        $path = storage_path('app/public/libros/' . $nombreArchivo);
        Log::info("📂 [RESUMEN-BOLETAS] Ruta archivo: {$path}");

        if (!file_exists($path)) {
            Log::error("❌ [RESUMEN-BOLETAS] Archivo no encontrado: {$path}");
            return response()->json(['error' => 'El archivo no existe: ' . $nombreArchivo], 404);
        }

        // Extraer rut y periodo desde el nombre de archivo
        if (preg_match('/RCV_RESUMEN_VENTA_(.*?)_(.*?)\.csv/', $nombreArchivo, $matches) !== 1) {
            Log::error("❌ [RESUMEN-BOLETAS] Nombre de archivo inválido: {$nombreArchivo}");
            return response()->json(['error' => 'Nombre de archivo inválido para resumen de boletas, formato esperado: RCV_RESUMEN_VENTA_{RUT}_{PERIODO}.csv'], 400);
        }

        $rutEmpresaConDV = $matches[1]; // en tu caso viene sin guion ni DV
        $periodo = $matches[2];

        // Normalizar rut a numerico + DV calculado
        $rutNumerico = preg_replace('/[^0-9]/', '', $rutEmpresaConDV);
        $dv = $this->calcularDV($rutNumerico);
        $rutEmpresaCompleto = $rutNumerico . $dv;
        Log::info("🆔 [RESUMEN-BOLETAS] RUT normalizado: {$rutEmpresaCompleto} | Periodo: {$periodo}");

        // Buscar empresa
        $empresa = ParEmpresa::whereRaw("REPLACE(rut, '-', '') = ?", [$rutEmpresaCompleto])->first();
        if (!$empresa) {
            Log::error("❌ [RESUMEN-BOLETAS] ParEmpresa no encontrada para RUT: {$rutEmpresaCompleto}");
            return response()->json(['error' => 'ParEmpresa no encontrada: ' . $rutEmpresaCompleto], 404);
        }
        $empresaId = $empresa->id;

        // Buscar periodo
        $tPeriodo = ParPeriodo::where('mes_ano', $periodo)->first();
        if (!$tPeriodo) {
            Log::error("❌ [RESUMEN-BOLETAS] Periodo no encontrado: {$periodo}");
            return response()->json(['error' => 'Periodo no encontrado: ' . $periodo], 404);
        }
        $periodoId = $tPeriodo->id;

        // Cliente fijo
        $rutCliente = '1111111-1';
        $rutClienteLimpio = str_replace('-', '', $rutCliente);
        $cliente = PosCliente::whereRaw("REPLACE(rut, '-', '') = ?", [$rutClienteLimpio])
            ->where('par_empresa_id', $empresaId)
            ->first();

        if (!$cliente) {
            $cliente = new PosCliente();
            $cliente->par_empresa_id = $empresaId;
            $cliente->rut = $rutCliente;
            $cliente->save();
            Log::info("👤 [RESUMEN-BOLETAS] Cliente placeholder creado ID={$cliente->id}");
        }
        $clienteId = $cliente->id;

        // Lectura CSV
        $file = fopen($path, 'r');
        if ($file === false) {
            Log::error("❌ [RESUMEN-BOLETAS] No se pudo abrir el archivo: {$path}");
            return response()->json(['error' => 'No se pudo abrir el archivo.'], 500);
        }

        $cabeceras = fgetcsv($file, 0, ';');
        Log::info("🧾 [RESUMEN-BOLETAS] Cabeceras detectadas:", $cabeceras ?: []);

        $map = [
            'Monto Exento' => 'Monto_Exento',
            'Monto Neto'   => 'Monto_Neto',
            'Monto IVA'    => 'Monto_IVA',
            'Monto Total'  => 'Monto_total',
        ];

        $indices = [];
        foreach ($map as $nombreCabecera => $campoSistema) {
            $index = is_array($cabeceras) ? array_search($nombreCabecera, $cabeceras) : false;
            if ($index !== false) {
                $indices[$campoSistema] = $index;
            } else {
                Log::warning("⚠️ [RESUMEN-BOLETAS] Cabecera '{$nombreCabecera}' no encontrada en CSV.");
            }
        }

        // Validación mínima de cabeceras
        foreach (['Monto_Exento','Monto_Neto','Monto_IVA','Monto_total'] as $c) {
            if (!isset($indices[$c])) {
                Log::warning("⚠️ [RESUMEN-BOLETAS] Índice faltante para {$c}. Se asumirá 0 en los montos.");
            }
        }

        $registros = [];
        $linea = 1;
        while (($row = fgetcsv($file, 0, ';')) !== false) {
            $linea++;
            if (!isset($row[0])) continue;

            $texto = trim($row[0]);

            // Detección flexible: acepta variantes, mayúsculas/minúsculas, con o sin "(39)/(48)"
            $esBoleta = preg_match('/Total\s*Oper.*Boleta\s*Electr/i', $texto);
            $esCPE    = preg_match('/Total\s*mes\s*Comprobantes\s*Pago\s*Electr/i', $texto);

            if (!$esBoleta && !$esCPE) {
                continue;
            }

            $fechaUltimoDia = Carbon::createFromFormat('Ym', $periodo)->endOfMonth()->toDateString();
            $tipoDoc = $esCPE ? 48 : 39;
            $estado = 0;

            $mExento = isset($indices['Monto_Exento']) ? $this->numero($row[$indices['Monto_Exento']] ?? '0') : 0;
            $mNeto   = isset($indices['Monto_Neto'])   ? $this->numero($row[$indices['Monto_Neto']]   ?? '0') : 0;
            $mIVA    = isset($indices['Monto_IVA'])    ? $this->numero($row[$indices['Monto_IVA']]    ?? '0') : 0;
            $mTotal  = isset($indices['Monto_total'])  ? $this->numero($row[$indices['Monto_total']]  ?? '0') : 0;

            $registros[] = [
                'rut_empresa'        => $rutEmpresaCompleto,
                'par_empresa_id'         => $empresaId,
                'periodo'            => $periodo,
                'par_periodo_id'       => $periodoId,
                'pos_cliente_id'     => $clienteId,
                'Rut_cliente'        => $rutCliente,
                'Razon_Social'       => 'Boletas',
                'Tipo_Doc'           => $tipoDoc,
                'Nro'                => 1,
                'Folio'              => 1,
                'Fecha_Docto'        => $fechaUltimoDia,
                'Fecha_Recepcion'    => $fechaUltimoDia,
                'Fecha_Acuse_Recibo' => $fechaUltimoDia,
                'Fecha_Reclamo'      => $fechaUltimoDia,
                'Monto_Exento'       => $mExento,
                'Monto_Neto'         => $mNeto,
                'Monto_IVA'          => $mIVA,
                'Monto_total'        => $mTotal,
                'estado'             => $estado
            ];

            Log::info("➕ [RESUMEN-BOLETAS] L{$linea} capturada: TipoDoc={$tipoDoc} Exento={$mExento} Neto={$mNeto} IVA={$mIVA} Total={$mTotal}");
        }
        fclose($file);

        Log::info("🔎 [RESUMEN-BOLETAS] Total filas válidas capturadas: " . count($registros));

        // Split por estado
        $registrosEstado0 = collect($registros)->where('estado', 0);
        $registrosEstado1 = collect($registros)->where('estado', 1);

        foreach ([39, 48] as $tipoDoc) {
            $registrosFiltrados = $registrosEstado0
                ->where('Tipo_Doc', $tipoDoc)
                ->where('par_empresa_id', $empresaId)
                ->where('rut_empresa', $rutEmpresaCompleto)
                ->where('periodo', $periodo);

            $cant = $registrosFiltrados->count();
            $sumaInsertados = [
                'Monto_Exento' => $registrosFiltrados->sum('Monto_Exento'),
                'Monto_Neto'   => $registrosFiltrados->sum('Monto_Neto'),
                'Monto_IVA'    => $registrosFiltrados->sum('Monto_IVA'),
                'Monto_total'  => $registrosFiltrados->sum('Monto_total'),
            ];

            Log::info("🧮 [RESUMEN-BOLETAS] TipoDoc={$tipoDoc} | filas={$cant} | CSV Exento={$sumaInsertados['Monto_Exento']} Neto={$sumaInsertados['Monto_Neto']} IVA={$sumaInsertados['Monto_IVA']} Total={$sumaInsertados['Monto_total']}");

            if ($cant === 0) {
                Log::info("ℹ️ [RESUMEN-BOLETAS] TipoDoc={$tipoDoc} no tiene filas para insertar/ajustar.");
                continue;
            }

            $sumaExistentes = ConSiiVenta::where('par_empresa_id', $empresaId)
                ->where('rut_empresa', $rutEmpresaCompleto)
                ->where('periodo', $periodo)
                ->where('Tipo_Doc', $tipoDoc)
                ->whereIn('estado', [0, 1])
                ->selectRaw('MAX(Nro) as max_nro, SUM(Monto_Exento) as Monto_Exento, SUM(Monto_Neto) as Monto_Neto, SUM(Monto_IVA) as Monto_IVA, SUM(Monto_total) as Monto_total')
                ->first();

            $existExento = $sumaExistentes->Monto_Exento ?? 0;
            $existNeto   = $sumaExistentes->Monto_Neto   ?? 0;
            $existIVA    = $sumaExistentes->Monto_IVA    ?? 0;
            $existTotal  = $sumaExistentes->Monto_total  ?? 0;

            Log::info("🏦 [RESUMEN-BOLETAS] TipoDoc={$tipoDoc} | BD Exento={$existExento} Neto={$existNeto} IVA={$existIVA} Total={$existTotal}");

            $difExento = $sumaInsertados['Monto_Exento'] - $existExento;
            $difNeto   = $sumaInsertados['Monto_Neto']   - $existNeto;
            $difIVA    = $sumaInsertados['Monto_IVA']    - $existIVA;
            $difTotal  = $sumaInsertados['Monto_total']  - $existTotal;

            Log::info("Δ [RESUMEN-BOLETAS] TipoDoc={$tipoDoc} | Dif Exento={$difExento} Neto={$difNeto} IVA={$difIVA} Total={$difTotal}");

            if ($difExento != 0 || $difNeto != 0 || $difIVA != 0 || $difTotal != 0) {
                $fechaUltimoDia = Carbon::createFromFormat('Ym', $periodo)->endOfMonth()->toDateString();

                $registroEstado1 = ConSiiVenta::where('par_empresa_id', $empresaId)
                    ->where('rut_empresa', $rutEmpresaCompleto)
                    ->where('periodo', $periodo)
                    ->where('Tipo_Doc', $tipoDoc)
                    ->where('estado', 1)
                    ->first();

                if ($registroEstado1) {
                    $registroEstado1->Monto_Exento += $difExento;
                    $registroEstado1->Monto_Neto   += $difNeto;
                    $registroEstado1->Monto_IVA    += $difIVA;
                    $registroEstado1->Monto_total  += $difTotal;
                    $registroEstado1->save();

                    Log::info("✏️ [RESUMEN-BOLETAS] UPDATE estado=1 TipoDoc={$tipoDoc} (ajuste aplicado)");
                } else {
                    $nuevoNro = ($sumaExistentes->max_nro ?? 1) + 1;

                    ConSiiVenta::create([
                        'rut_empresa'        => $rutEmpresaCompleto,
                        'par_empresa_id'         => $empresaId,
                        'periodo'            => $periodo,
                        'par_periodo_id'       => $periodoId,
                        'pos_cliente_id'     => $clienteId,
                        'Rut_cliente'        => $rutCliente,
                        'Razon_Social'       => 'Boletas',
                        'Tipo_Doc'           => $tipoDoc,
                        'Nro'                => $nuevoNro,
                        'Folio'              => $nuevoNro,
                        'Fecha_Docto'        => $fechaUltimoDia,
                        'Fecha_Recepcion'    => $fechaUltimoDia,
                        'Fecha_Acuse_Recibo' => $fechaUltimoDia,
                        'Fecha_Reclamo'      => $fechaUltimoDia,
                        'Monto_Exento'       => $difExento,
                        'Monto_Neto'         => $difNeto,
                        'Monto_IVA'          => $difIVA,
                        'Monto_total'        => $difTotal,
                        'estado'             => 1
                    ]);

                    Log::info("🆕 [RESUMEN-BOLETAS] CREATE estado=1 TipoDoc={$tipoDoc} (ajuste creado Nro={$nuevoNro})");
                }
            } else {
                Log::info("✅ [RESUMEN-BOLETAS] TipoDoc={$tipoDoc} sin diferencias. No se realizan ajustes.");
            }
        }

        // Si tuvieras registros con estado=1 precargados en el CSV (no es el caso aquí),
        // se actualizarían con este bloque:
        foreach ($registrosEstado1 as $registro) {
            ConSiiVenta::updateOrCreate(
                [
                    'par_empresa_id' => $registro['par_empresa_id'],
                    'rut_empresa' => $registro['rut_empresa'],
                    'periodo' => $registro['periodo'],
                    'Tipo_Doc' => $registro['Tipo_Doc'],
                    'estado' => 1
                ],
                array_merge($registro, ['par_periodo_id' => $periodoId])
            );
        }

        Log::info("🏁 [RESUMEN-BOLETAS] Proceso finalizado. Filas capturadas: " . count($registros));
        return response()->json([
            'mensaje' => 'Resumen de boletas y comprobantes importado correctamente',
            'total_insertados' => count($registros)
        ]);

    } catch (\Exception $e) {
        Log::error("❌ [RESUMEN-BOLETAS] Excepción: {$e->getMessage()} en línea {$e->getLine()}");
        return response()->json([
            'error' => 'Ocurrió un error al importar el resumen',
            'detalle' => $e->getMessage(),
            'linea' => $e->getLine()
        ], 500);
    }
}





private function calcularDV($rut)
{
    $s = 1;
    $m = 0;
    for (; $rut; $rut = floor($rut / 10))
        $s = ($s + $rut % 10 * (9 - $m++ % 6)) % 11;

    return $s ? chr($s + 47) : 'K';
}










    public function subirLibro(Request $request)
    {
        if (!$request->hasFile('archivo')) {
            return response()->json(['error' => 'No se recibió ningún archivo.'], 400);
        }

        $archivo = $request->file('archivo');

        // Guardar en storage/app/libros/
        $nombreArchivo = $archivo->getClientOriginalName();
        $archivo->storeAs('public/libros', $nombreArchivo);

        // Llamar a la importación
        return $this->importarVentasDesdeCSV($nombreArchivo);
    }

    private function buscarValor($cabeceras, $row, $nombreColumna)
    {
        $index = array_search($nombreColumna, $cabeceras);

        // SOLO PARA DEBUG: mostrar si encontró el índice o no
        if ($index === false) {
            Log::info("No se encontró la columna: '$nombreColumna' en el archivo CSV.");
        } else {
            Log::info("Encontrada columna '$nombreColumna' en índice: $index con valor: " . ($row[$index] ?? 'NO VALOR'));
        }

        if ($index !== false && isset($row[$index])) {
            $valor = trim($row[$index]);
            return $valor !== '' ? $valor : null;
        }
        return null;
    }

    private function formatearFecha($fecha)
    {
        if (!$fecha) return null;

        $fecha = trim($fecha); // 🔥 IMPORTANTE

        if (strpos($fecha, '/') === false) {
            return null;
        }

        [$dia, $mes, $anio] = explode('/', $fecha);

        if (!is_numeric($dia) || !is_numeric($mes) || !is_numeric($anio)) {
            return null;
        }

        if (!checkdate((int)$mes, (int)$dia, (int)$anio)) {
            return null;
        }

        return "$anio-$mes-$dia";
    }

    private function formatearFechaConHora($fecha)
    {
        if (!$fecha) return null;

        // Separar fecha y hora si existe
        $partes = explode(' ', trim($fecha));
        $soloFecha = $partes[0];
        $hora = $partes[1] ?? '00:00:00';

        if (strpos($soloFecha, '/') === false) {
            return null;
        }

        [$dia, $mes, $anio] = explode('/', $soloFecha);

        if (!is_numeric($dia) || !is_numeric($mes) || !is_numeric($anio)) {
            return null;
        }

        if (!checkdate((int)$mes, (int)$dia, (int)$anio)) {
            return null;
        }

        return sprintf('%04d-%02d-%02d %s', $anio, $mes, $dia, $hora);
    }


    private function numero($valor)
    {
        // Limpia y convierte a número. Si no es numérico, retorna 0.
        if (is_null($valor)) return 0;
        $limpio = str_replace(['$', '.', ' ', ','], '', $valor);
        return is_numeric($limpio) ? (float)$limpio : 0;
    }

public function descargar(Request $request)
{
    // Validación defensiva: evitamos que la regla 'exists' lance una excepción
    // si la tabla no existe en algunos entornos. Usamos comprobaciones seguras
    // mediante los modelos y capturamos QueryException.
    $validator = Validator::make($request->all(), [
        'par_empresa_id' => 'required|integer', 
        'periodo' => 'required|integer'
    ]);

    $validator->after(function ($v) use ($request) {
        try {
            if (!\App\Models\Parametros\ParEmpresa::where('id', $request->par_empresa_id)->exists()) {
                $v->errors()->add('par_empresa_id', 'La empresa indicada no existe.');
            }
        } catch (\Exception $ex) {
            // Si la consulta falla por ausencia de tabla, lo registramos y agregamos error
            Log::warning('Validación: no se pudo verificar existencia de par_empresas: ' . $ex->getMessage());
            $v->errors()->add('par_empresa_id', 'No se pudo verificar la empresa en la base de datos.');
        }

            try {
                // Soportar ambos esquemas: par_periodos o par_periodos según migración
                $periodoExists = false;
                try {
                    $periodoExists = \App\Models\Parametros\ParPeriodo::where('id', $request->periodo)->exists();
                } catch (\Exception $_) {
                    // fallback a tabla antigua si aplica
                    $periodoExists = DB::table('par_periodos')->where('id', $request->periodo)->exists();
                }

            if (!$periodoExists) {
                $v->errors()->add('periodo', 'El período indicado no existe.');
            }
        } catch (\Exception $ex) {
            Log::warning('Validación: no se pudo verificar existencia de periodos: ' . $ex->getMessage());
            $v->errors()->add('periodo', 'No se pudo verificar el periodo en la base de datos.');
        }
    });

    if ($validator->fails()) {
        return response()->json(['errors' => $validator->errors()], 422);
    }

    try {
 
      

        // 📦 Obtener empresa y rut con guión
        $empresa = \App\Models\Parametros\ParEmpresa::findOrFail($request->par_empresa_id);
        $rutLimpio = preg_replace('/[^0-9kK]/', '', $empresa->rut);
        $rut = substr($rutLimpio, 0, -1) . '-' . strtoupper(substr($rutLimpio, -1));

        // 🔓 Desencriptar clave SII
        $clave = Crypt::decryptString($empresa->sii);

        // 📅 Separar mes y año del periodo
        $periodo = \App\Models\Parametros\ParPeriodo::findOrFail($request->periodo);
        [$mes, $anio] = explode('-', $periodo->nombre);

        $token = $request->bearerToken();

        Log::info('📥 Ejecutando script Node.js con:', [
            'rut' => $rut,
            'clave' => '[oculto]',
            'mes' => $mes,
            'anio' => $anio,
        ]);

        // 🧠 Pasar todos los datos necesarios al script, incluyendo el token
        $comando = [
            'env',
            'NODE_PATH=/opt/node_modules/lib/node_modules',
            '/usr/bin/node',
            'scripts/descargar_ventas_sii.js',
            $rut,
            $clave,
            $mes,
            $anio,
            $request->par_empresa_id,
            $request->periodo,
            $token
        ];

        $process = new Process($comando);
        $process->setWorkingDirectory(base_path());
        $process->setTimeout(300); // ⏱️ Tiempo máximo de espera (5 minutos)
        $process->run();

        if (!$process->isSuccessful()) {
            throw new ProcessFailedException($process);
        }

        $output = $process->getOutput();
        Log::info("📤 Salida del script Node.js:", [$output]);

        return response()->json([
            'message' => '✅ Script ejecutado correctamente.',
            'salida' => $output,
        ]);
    } catch (\Throwable $e) {
        Log::error('❌ Error al ejecutar script Node.js: ' . $e->getMessage());

        return response()->json([
            'error' => '❌ Fallo al ejecutar el script Node.js',
            'detalle' => $e->getMessage(),
        ], 500);
    }
}



}