<?php

namespace App\Http\Controllers\Contabilidad;

use App\Models\Contabilidad\ConCuenta;
use App\Models\Contabilidad\ConSubCuenta;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;

class ConSubCuentaController extends Controller
{
    public function index()
    {
        return ConSubCuenta::with(['ConCuenta', 'ConCuenta.ConCuentaMadre.ConCategoriaCuenta', 'ConCuenta.ConCuentaMadre.ConCategoriaCuenta.tipoCuenta', 'ConCuenta.ConPAuxiliar'])->where('estado', 1)->get();
    }

    public function pagoProveedores()
    {
        return ConSubCuenta::with([
            'ConCuenta',
            'ConCuenta.ConCuentaMadre.ConCategoriaCuenta.tipoCuenta',
            'ConCuenta.ConPAuxiliar'
        ])
            ->whereHas('ConCuenta', function ($q) {
                $q->where('con_cuenta_madre_id', 1)
                    ->orWhere('con_cuenta_madre_id', 4);
            })
            ->where('estado', 1)
            ->get();
    }

    public function store(Request $request)
    {
        return DB::transaction(function () use ($request) {
            $cuenta = ConCuenta::findOrFail($request->con_cuenta_id);
            $codigoBase = $cuenta->codigo;
            $ultimoSubCodigo = ConSubCuenta::where('codigo', 'like', $codigoBase . '%')
                ->orderByRaw('CAST(SUBSTRING(codigo, ' . (strlen($codigoBase) + 1) . ') AS UNSIGNED) DESC')
                ->lockForUpdate()
                ->value('codigo');

            $ultimoSubCorrelativo = $ultimoSubCodigo ? intval(substr($ultimoSubCodigo, strlen($codigoBase))) : 0;
            $nuevoSubCorrelativo = str_pad($ultimoSubCorrelativo + 1, 3, '0', STR_PAD_LEFT);
            $nuevoCodigo = $codigoBase . $nuevoSubCorrelativo;

            $ConSubCuenta = new ConSubCuenta();
            $ConSubCuenta->con_cuenta_id = $request->con_cuenta_id;
            $ConSubCuenta->nombre = $request->nombre;
            $ConSubCuenta->especial = $request->especial;
            $ConSubCuenta->codigo = $nuevoCodigo;
            $ConSubCuenta->save();

            return response()->json($ConSubCuenta, 201);
        });
    }

    public function show($id)
    {
        return ConSubCuenta::findOrFail($id);
    }

    public function update(Request $request, $id)
    {
        $subCuenta = ConSubCuenta::findOrFail($id);

        $validated = $request->validate([
            'nombre' => 'nullable|string|max:255',
            'con_cuenta_id' => 'nullable|integer',
        ]);

        $subCuenta->update($validated);

        return response()->json($subCuenta);
    }

    public function destroy(ConSubCuenta $conSubCuenta)
    {
        try {
            $conSubCuenta->estado = 0;
            $conSubCuenta->save();

            return response()->json(['message' => 'Subcuenta eliminada correctamente.'], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Error al eliminar la subcuenta.'], 500);
        }
    }
}
