<?php

namespace App\Http\Controllers\Contabilidad;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB; // esta línea importa DB correctamente
use App\Models\Contabilidad\ConVoucher;
use App\Models\Contabilidad\ConLibroDiario;
use App\Models\Contabilidad\ConFolioVoucher;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;

class ConVoucherController extends Controller
{
    public function index()
    {
        return ConVoucher::with(['libroDiarios', 'conTVouchers', 'libroDiarios.subCuenta'])->get();
    }

public function Filtro(Request $request)
{
    $filtro = $request->filtro;
    $periodo = $request->periodo !== 'null' ? $request->periodo : null;
    $anio = $request->anio !== 'null' ? $request->anio : null;

        return ConVoucher::with(['libroDiarios', 'conTVouchers', 'libroDiarios.subCuenta'])
        ->where('par_empresa_id', $filtro)
        ->when($periodo, function ($query) use ($periodo) {
            $query->where('par_periodo_id', $periodo);
        })
        ->when($anio, function ($query) use ($anio) {
            $query->where('par_anio_id', $anio);
        })
        ->where('estado', 1)
        ->get();
}

    public function Balance(Request $request)
    {
        $filtro = $request->filtro;
        $periodo = $request->periodo !== 'null' ? $request->periodo : null;
        $anio = $request->anio !== 'null' ? $request->anio : null;

        $vouchers = ConVoucher::with([
            'conTVouchers',
            'libroDiarios.subCuenta.conCuenta.conCuentaMadre.conCategoriaCuenta.tipoCuenta',
        ])
            ->where('par_empresa_id', $filtro)
            ->when($periodo, function ($query) use ($periodo) {
                $query->where('par_periodo_id', $periodo);
            })
            ->when($anio, function ($query) use ($anio) {
                $query->where('par_anio_id', $anio);
            })
            ->where('estado', 1)
            ->get();

        $movimientos = [];

        foreach ($vouchers as $voucher) {
            foreach ($voucher->libroDiarios as $mov) {
                $subCuenta = $mov->subCuenta;
                $cuenta = $subCuenta->conCuenta;
                $cuentaMadre = $cuenta->conCuentaMadre;
                $categoria = $cuentaMadre->conCategoriaCuenta;
                $tipoCuenta = $categoria->tipoCuenta;

                $key = $subCuenta->codigo;

                if (!isset($movimientos[$key])) {
                    $movimientos[$key] = [
                        'codigo'      => $subCuenta->codigo,
                        'nombre'      => $subCuenta->nombre,
                        'tipo_cuenta' => strtoupper($tipoCuenta->nombre),
                        'debe'        => 0,
                        'haber'       => 0,
                    ];
                }

                $movimientos[$key]['debe']  += floatval($mov->debe);
                $movimientos[$key]['haber'] += floatval($mov->haber);
            }
        }

        $totales = [
            'debe' => 0,
            'haber' => 0,
            'deudor' => 0,
            'acreedor' => 0,
            'activo' => 0,
            'pasivo' => 0,
            'perdida' => 0,
            'ganancia' => 0,
        ];

        $balance = collect($movimientos)->map(function ($item) use (&$totales) {
            $debe = floatval($item['debe']);
            $haber = floatval($item['haber']);
            $tipoCuenta = strtoupper($item['tipo_cuenta']);

            $deudor = $debe > $haber ? $debe - $haber : 0;
            $acreedor = $haber > $debe ? $haber - $debe : 0;

            $activo = $pasivo = $perdida = $ganancia = 0;

            if ($tipoCuenta === 'ACTIVO' && ($debe > 0 || $haber > 0)) {
                if ($debe >= $haber) {
                    $activo = $debe - $haber;
                } else {
                    $pasivo = $haber - $debe;
                }
            }

            if ($tipoCuenta === 'PASIVO' && ($debe > 0 || $haber > 0)) {
                if ($haber >= $debe) {
                    $pasivo = $haber - $debe;
                } else {
                    $activo = $debe - $haber;
                }
            }

            if ($tipoCuenta === 'RESULTADO') {
                if ($deudor > 0) {
                    $perdida = $deudor;
                } elseif ($acreedor > 0) {
                    $ganancia = $acreedor;
                }
            }

            $totales['debe']     += $debe;
            $totales['haber']    += $haber;
            $totales['deudor']   += $deudor;
            $totales['acreedor'] += $acreedor;
            $totales['activo']   += $activo;
            $totales['pasivo']   += $pasivo;
            $totales['perdida']  += $perdida;
            $totales['ganancia'] += $ganancia;

            return [
                'codigo'   => $item['codigo'],
                'nombre'   => $item['nombre'],
                'debe'     => $debe,
                'haber'    => $haber,
                'deudor'   => $deudor,
                'acreedor' => $acreedor,
                'activo'   => $activo,
                'pasivo'   => $pasivo,
                'perdida'  => $perdida,
                'ganancia' => $ganancia,
            ];
        })->values();

        $resultadoDelEjercicio = [
            'gananciaResultado' => max($totales['ganancia'] - $totales['perdida'], 0),
            'perdidaResultado' => max($totales['perdida'] - $totales['ganancia'], 0),
            'perdidaBalance' => max($totales['pasivo'] - $totales['activo'], 0),
            'gananciaBalance' => max($totales['activo'] - $totales['pasivo'], 0),
        ];

        return response()->json([
            'balance' => $balance,
            'totales' => $totales,
            'resultadoDelEjercicio' => $resultadoDelEjercicio,
        ]);
    }

    public function store(Request $request)
    {
        $data = $request->json()->all();

        Log::info('\ud83d\udce6 REQUEST JSON completo:', $data);

        $list = collect($data['list'] ?? [])->filter()->values();
        $userId = $data['user_id'] ?? null;
        $glosa = $data['glosa'] ?? null;
        $tipoVoucher = $data['con_t_voucher_id'] ?? 1;
        $irCompras = $data['ir_compras'] ?? 0;
        $voucherReferenciaId = $data['voucher_referencia_id'] ?? null;

        $fecha = $data['fecha'] ?? now();
        try {
            $fecha = \Carbon\Carbon::parse($fecha);
        } catch (\Exception $e) {
            Log::warning('\u26a0\ufe0f Fecha inválida recibida. Usando fecha actual.', ['fecha_recibida' => $fecha]);
            $fecha = now();
        }

        Log::info('\ud83d\udce5 Datos clave recibidos:', [
            'glosa' => $glosa,
            'tipo_voucher' => $tipoVoucher,
            'ir_compras' => $irCompras,
            'user_id' => $userId,
            'fecha' => $fecha->toDateString(),
            'con_voucher_p_id' => $voucherReferenciaId,
        ]);

        if ($list->isEmpty()) {
            Log::error('\u274c Lista vacía o inválida');
            return response()->json(['mensaje' => 'No se recibieron datos válidos'], 400);
        }

        if (!$userId) {
            Log::error('\u274c user_id no recibido');
            return response()->json(['mensaje' => 'Falta user_id'], 400);
        }

    $empresaId = $list[0]['empresa_id'];
    $periodoId = $list[0]['par_periodo_id'] ?? $list[0]['t_periodo_id'];

        $anioId = DB::table('t_periodos')->where('id', $periodoId)->value('par_anio_id');

        try {
            DB::beginTransaction();

            $folioData = DB::table('con_folio_vouchers')
                ->where('par_empresa_id', $empresaId)
                ->where('con_t_voucher_id', $tipoVoucher)
                ->where('par_anio_id', $anioId)
                ->first();

            $nuevoFolio = $folioData ? $folioData->folio + 1 : 1;

            if (!$folioData) {
                DB::table('con_folio_vouchers')->insert([
                    'par_empresa_id' => $empresaId,
                    'par_anio_id' => $anioId,
                    'con_t_voucher_id' => $tipoVoucher,
                    'folio' => 1,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            } else {
                DB::table('con_folio_vouchers')
                    ->where('par_empresa_id', $empresaId)
                    ->where('con_t_voucher_id', $tipoVoucher)
                    ->where('par_anio_id', $anioId)
                    ->update(['folio' => $nuevoFolio]);
            }

            $voucherId = DB::table('con_vouchers')->insertGetId([
                'par_empresa_id' => $empresaId,
                'par_anio_id' => $anioId,
                'par_periodo_id' => $periodoId,
                'con_t_voucher_id' => $tipoVoucher,
                'folio' => $nuevoFolio,
                'glosa' => $glosa,
                'user_id' => $userId,
                'fecha' => $fecha,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            $agrupados = [];

            foreach ($list as $item) {
                $key = $item['con_sub_cuenta_id'];

                if (!isset($agrupados[$key])) {
                    $agrupados[$key] = [
                        'con_voucher_id'    => $voucherId,
                        'par_empresa_id'        => $item['empresa_id'],
                        'par_anio_id'           => $anioId,
                        'par_periodo_id'      => $periodoId,
                        'con_sub_cuenta_id' => $key,
                        'debe'              => 0,
                        'haber'             => 0,
                        'created_at'        => now(),
                        'updated_at'        => now(),
                    ];
                }

                $agrupados[$key]['debe'] += floatval($item['debe'] ?? $item['total_debe'] ?? 0);
                $agrupados[$key]['haber'] += floatval($item['haber'] ?? $item['total_haber'] ?? 0);
            }

            DB::table('con_libro_diarios')->insert(array_values($agrupados));

            $idsActualizar = array_column($list->toArray(), 'id');

            if ($irCompras == 1) {
                DB::table('con_pre_lib_compras')
                    ->whereIn('id', $idsActualizar)
                    ->update([
                        'con_voucher_id' => $voucherId,
                        'con_voucher_p_id' => $voucherReferenciaId,
                        'estado' => 0,
                        'updated_at' => now()
                    ]);
            }

            if ($irCompras == 2) {
                DB::table('con_pre_lib_ventas')
                    ->whereIn('id', $idsActualizar)
                    ->update([
                        'con_voucher_id' => $voucherId,
                        'con_voucher_p_id' => $voucherReferenciaId,
                        'estado' => 0,
                        'updated_at' => now()
                    ]);
            }

            if ($irCompras == 3) {
                DB::table('con_pre_lib_h_compras')
                    ->whereIn('id', $idsActualizar)
                    ->update([
                        'con_voucher_id' => $voucherId,
                        'con_voucher_p_id' => $voucherReferenciaId,
                        'estado' => 0,
                        'updated_at' => now()
                    ]);
            }

            DB::commit();

            return response()->json([
                'mensaje' => 'Voucher creado correctamente',
                'voucher_id' => $voucherId,
                'folio' => $nuevoFolio
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('\u274c Error al guardar voucher y libro diario', ['error' => $e->getMessage()]);
            return response()->json(['mensaje' => 'Error interno al guardar el voucher.'], 500);
        }
    }

    public function show($id)
    {
        return ConVoucher::findOrFail($id);
    }

    public function update(Request $request, $id)
    {
        $voucher = ConVoucher::findOrFail($id);

        $voucher->update($request->all());

        return response()->json($voucher);
    }

    public function destroy(ConVoucher $conVoucher)
    {
        DB::beginTransaction();

        try {
            $conVoucher->estado = 0;
            $conVoucher->save();

            $voucherId = $conVoucher->id;

            DB::table('con_libro_diarios')
                ->where('con_voucher_id', $voucherId)
                ->update(['estado' => 1]);

            DB::table('con_pre_lib_ventas')
                ->where('con_voucher_id', $voucherId)
                ->update(['estado' => 1]);

            DB::table('con_pre_lib_compras')
                ->where('con_voucher_id', $voucherId)
                ->update(['estado' => 1]);

            DB::commit();

            return response()->json(['message' => 'Voucher desactivado y registros restaurados correctamente.']);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al desactivar voucher: ' . $e->getMessage());

            return response()->json([
                'message' => 'Error al desactivar el voucher.',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
