<?php

namespace App\Http\Controllers;
    use App\Models\Pos\PosArriendoDetalle;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use App\Models\Pos\PosMDespacho;
use Illuminate\Http\Request;
use App\Models\Pos\PosPosArriendoDetalle;
use Illuminate\Support\Facades\DB;
use App\Models\Pos\PosMantencionDetalle;

class DMantencionesController extends Controller
{

public function maquinariasActivas(Request $request)
{
    $empresaId = $request->empresa;

    if (!$empresaId) {
        return response()->json(['error' => 'Falta filtro de empresa'], 400);
    }

    // Obtener el RUT de la empresa
    $empresa = \App\Models\Parametros\ParEmpresa::find($empresaId);
    if (!$empresa) {
        return response()->json(['error' => 'Empresa no encontrada'], 404);
    }

    $pos_maquinarias = \App\Models\Pos\PosMaquinaria::whereHas('posCliente', function ($q) use ($empresa) {
        $q->where('rut', $empresa->rut);
    })
    ->where('estado', 1) // Solo pos_maquinarias activas
    ->where('par_empresa_id', $empresaId)
    ->select('id','codigo', 'nombre')
    ->orderBy('nombre')
    ->get();


    return response()->json($pos_maquinarias);
}


public function filtroAvanzado(Request $request)
{
    $empresaId = $request->empresa;

    if (!$empresaId) {
        return response()->json(['error' => 'Falta filtro de empresa'], 400);
    }

    // Obtener el RUT de la empresa
    $empresa = \App\Models\Parametros\ParEmpresa::find($empresaId);
    if (!$empresa) {
        return response()->json(['error' => 'Empresa no encontrada'], 404);
    }

    // ✅ Solo obtener filtro de pos_maquinarias
    $pos_maquinarias = $request->input('posMaquinaria', []); // Array de IDs de pos_maquinarias

    // Query base
    $query = PosMantencionDetalle::whereHas('posMaquinaria.posCliente', function ($q) use ($empresa) {
        $q->where('rut', $empresa->rut);
    })
    ->where('estado', 0);

    // ✅ Solo filtro por pos_maquinarias (si se envía y no está vacío)
    if (!empty($pos_maquinarias) && is_array($pos_maquinarias)) {
        $maquinariasIds = array_filter(array_map('intval', $pos_maquinarias));
        if (!empty($maquinariasIds)) {
            $query->whereIn('pos_maquinaria_id', $maquinariasIds);
        }
    }

    // ✅ Solo obtener suma y total usando agregaciones
    $resultado = $query->select(
        DB::raw('COUNT(*) as total_mantenciones'),
        DB::raw('SUM(valor) as suma_valor_total')
    )->first();

    return response()->json([
        'total_mantenciones' => $resultado->total_mantenciones,
        'suma_valor_total' => $resultado->suma_valor_total ?: 0
    ]);
}

public function fAvanzadoManteciones(Request $request)
{
    $empresaId = $request->empresa;

    if (!$empresaId) {
        return response()->json(['error' => 'Falta filtro de empresa'], 400);
    }

    // Obtener el RUT de la empresa
    $empresa = \App\Models\Parametros\ParEmpresa::find($empresaId);
    if (!$empresa) {
        return response()->json(['error' => 'Empresa no encontrada'], 404);
    }

    // ✅ Solo obtener filtro de pos_maquinarias
    $pos_maquinarias = $request->input('posMaquinaria', []); // Array de IDs de pos_maquinarias

    // Query base
    $query = PosMantencionDetalle::with(['posMaquinaria'])
        ->whereHas('posMaquinaria.posCliente', function ($q) use ($empresa) {
            $q->where('rut', $empresa->rut);
        })
        ->where('estado', 0);

    // ✅ Solo filtro por pos_maquinarias (si se envía y no está vacío)
    if (!empty($pos_maquinarias) && is_array($pos_maquinarias)) {
        $maquinariasIds = array_filter(array_map('intval', $pos_maquinarias));
        if (!empty($maquinariasIds)) {
            $query->whereIn('pos_maquinaria_id', $maquinariasIds);
        }
    }

    // ✅ Obtener el listado completo de posMantenciones con sus valores
    $posMantenciones = $query->select([
            'id',
            'pos_maquinaria_id', 
            'valor',
            'f_entrada',
            'f_salida',
            'detalle',
            'estado'
        ])
        ->orderBy('f_salida', 'desc')
        ->get();

    // ✅ Calcular totales para información adicional
    $totales = [
        'total_mantenciones' => $posMantenciones->count(),
        'suma_valor_total' => $posMantenciones->sum('valor')
    ];

    return response()->json([
        'posMantenciones' => $posMantenciones,
        'totales' => $totales
    ]);
}

public function fAvanzadoMDetalles(Request $request)
{
    $empresaId = $request->empresa;

    if (!$empresaId) {
        return response()->json(['error' => 'Falta filtro de empresa'], 400);
    }

    // Obtener el RUT de la empresa
    $empresa = \App\Models\Parametros\ParEmpresa::find($empresaId);
    if (!$empresa) {
        return response()->json(['error' => 'Empresa no encontrada'], 404);
    }

    // ✅ Solo obtener filtro mdetalle
    $mdetalle = $request->input('mdetalle', []); // ✅ Filtro para pos_mantencion_detalle_id

    // Query base - usando MDMantencione sin cargar relación inexistente
    $query = \App\Models\Pos\PosMDMantencione::with(['posMaquinaria']);

    // ✅ Solo filtro por pos_mantencion_detalle_id (si se envía y no está vacío)
    if (!empty($mdetalle) && is_array($mdetalle)) {
        $mdetalleIds = array_filter(array_map('intval', $mdetalle));
        if (!empty($mdetalleIds)) {
            $query->whereIn('pos_mantencion_detalle_id', $mdetalleIds);
        }
    }

    // ✅ Obtener el listado completo de detalles de posMantenciones
    $detallesMantenciones = $query->select([
            'id',
            'pos_mantencione_id',
            'pos_mantencion_detalle_id',
            'pos_maquinaria_id', 
            'detalle',
            'cantidad',
            'total',
            'valor'
        ])
        ->orderBy('id', 'desc')
        ->get();

    // ✅ Calcular totales para información adicional
    $totales = [
        'total_registros' => $detallesMantenciones->count(),
        'suma_cantidad_total' => $detallesMantenciones->sum('cantidad'),
        'suma_total_general' => $detallesMantenciones->sum('total'),
        'suma_valor_total' => $detallesMantenciones->sum('valor')
    ];

    return response()->json([
        'detalles_mantenciones' => $detallesMantenciones,
        'totales' => $totales
    ]);
}

public function fAvanzadoArriendos(Request $request)
{
    $empresaId = $request->empresa;

    if (!$empresaId) {
        return response()->json(['error' => 'Falta filtro de empresa'], 400);
    }

    // Obtener el RUT de la empresa
    $empresa = \App\Models\Parametros\ParEmpresa::find($empresaId);
    if (!$empresa) {
        return response()->json(['error' => 'Empresa no encontrada'], 404);
    }

    // ✅ Solo obtener filtro de pos_maquinarias
    $pos_maquinarias = $request->input('posMaquinaria', []); // Array de IDs de pos_maquinarias

    // Query base - ahora usando ArriendoDetalle
    $query = PosArriendoDetalle::whereHas('posMaquinaria.posCliente', function ($q) use ($empresa) {
        $q->where('rut', $empresa->rut);
    })
    ->where('estado', 0); // Asumiendo que estado 0 son arriendos finalizados

    // ✅ Solo filtro por pos_maquinarias (si se envía y no está vacío)
    if (!empty($pos_maquinarias) && is_array($pos_maquinarias)) {
        $maquinariasIds = array_filter(array_map('intval', $pos_maquinarias));
        if (!empty($maquinariasIds)) {
            $query->whereIn('pos_maquinaria_id', $maquinariasIds);
        }
    }

    // ✅ Obtener suma de v_total y cantidad usando agregaciones
    $resultado = $query->select(
        DB::raw('COUNT(*) as total_arriendos'),
        DB::raw('SUM(v_total) as suma_v_total'),
        DB::raw('SUM(cantidad) as suma_cantidad')
    )->first();

    return response()->json([
        'total_arriendos' => $resultado->total_arriendos,
        'suma_v_total' => $resultado->suma_v_total ?: 0,
        'suma_cantidad' => $resultado->suma_cantidad ?: 0
    ]);
}


public function fAvanzadoMaquinarias(Request $request)
{
    $empresaId = $request->empresa;

    if (!$empresaId) {
        return response()->json(['error' => 'Falta filtro de empresa'], 400);
    }

    // Obtener el RUT de la empresa
    $empresa = \App\Models\Parametros\ParEmpresa::find($empresaId);
    if (!$empresa) {
        return response()->json(['error' => 'Empresa no encontrada'], 404);
    }

    // ✅ Solo obtener filtro de pos_maquinarias
    $pos_maquinarias = $request->input('posMaquinaria', []); // Array de IDs de pos_maquinarias

    // Query base - ahora usando posMaquinaria directamente
    $query = \App\Models\Pos\PosMaquinaria::whereHas('posCliente', function ($q) use ($empresa) {
        $q->where('rut', $empresa->rut);
    })
    ->where('estado', 1); // Solo pos_maquinarias activas

    // ✅ Solo filtro por pos_maquinarias (si se envía y no está vacío)
    if (!empty($pos_maquinarias) && is_array($pos_maquinarias)) {
        $maquinariasIds = array_filter(array_map('intval', $pos_maquinarias));
        if (!empty($maquinariasIds)) {
            $query->whereIn('id', $maquinariasIds);
        }
    }

    // ✅ Obtener suma de p_compra usando agregaciones
    $resultado = $query->select(
        DB::raw('COUNT(*) as total_maquinarias'),
        DB::raw('SUM(p_compra) as suma_p_compra')
    )->first();

    return response()->json([
        'total_maquinarias' => $resultado->total_maquinarias,
        'suma_p_compra' => $resultado->suma_p_compra ?: 0
    ]);
}


public function MesEnCurso(Request $request)
{
    $empresaId = $request->empresa;
    $anio = $request->anio;
    $mes = $request->mes;

    if (!$empresaId || !$anio || !$mes) {
        return response()->json(['error' => 'Faltan filtros necesarios'], 400);
    }

    $resumen = PosArriendoDetalle::whereHas('arriendo', function ($q) use ($empresaId) {
        $q->where('par_empresa_id', $empresaId);
    })
    ->whereYear('f_salida', $anio)
    ->whereMonth('f_salida', $mes)
    ->select(
        DB::raw('MONTH(f_salida) as mes'),
        DB::raw('COUNT(*) as cantidad_arriendos'),
        DB::raw('SUM(cantidad) as total_dias'),
        DB::raw('SUM(v_total) as suma_v_total'),
        DB::raw('ROUND(AVG(v_total / cantidad), 2) as promedio_por_dia')
    )
    ->groupBy(DB::raw('MONTH(f_salida)'))
    ->orderBy(DB::raw('MONTH(f_salida)'))
    ->get();

    return response()->json($resumen);
    }


public function MesEnCursoMantencion(Request $request)
{
    $empresaId = $request->empresa;
    $anio = $request->anio;
    $mes = $request->mes;

    if (!$empresaId) {
        return response()->json(['error' => 'Falta filtro de empresa'], 400);
    }

    // Obtener el RUT de la empresa
    $empresa = \App\Models\Parametros\ParEmpresa::find($empresaId);
    if (!$empresa) {
        return response()->json(['error' => 'Empresa no encontrada'], 404);
    }

    $baseQuery = PosMantencionDetalle::whereHas('posMaquinaria.posCliente', function ($q) use ($empresa) {
        $q->whereNot('rut', $empresa->rut);
    });

    // Siempre obtener posMantenciones activas (estado 1)
    $mantencionesActivas = $baseQuery->where('estado', 1)
        ->select(
            DB::raw('COUNT(*) as cantidad_mantenciones_activas'),
            DB::raw('COUNT(DISTINCT pos_maquinaria_id) as cantidad_maquinarias_unicas'),
            DB::raw('SUM(valor) as suma_valor_total_activas')
        )
        ->first();

    $resultado = [
        'mantenciones_activas' => $mantencionesActivas
    ];

    // Si hay año y mes, agregar resumen mensual
    if ($anio && $mes) {
        $resumenMensual = PosMantencionDetalle::whereHas('posMaquinaria.posCliente', function ($q) use ($empresa) {
            $q->whereNot('rut', $empresa->rut);
        })
        ->whereYear('f_entrada', $anio)
        ->whereMonth('f_entrada', $mes)
        ->where('estado', 0)
        ->select(
            DB::raw('MONTH(f_entrada) as mes'),
            DB::raw('COUNT(*) as cantidad_mantenciones'),
            DB::raw('SUM(valor) as suma_valor_total'),
            DB::raw('ROUND(AVG(valor), 2) as promedio_valor_mantencion')
        )
        ->groupBy(DB::raw('MONTH(f_entrada)'))
        ->orderBy(DB::raw('MONTH(f_entrada)'))
        ->first();

        $resultado['resumen_mensual'] = $resumenMensual;
    }

    return response()->json($resultado);
}

public function MesEnCursoManPropias(Request $request)
{
    $empresaId = $request->empresa;
    $anio = $request->anio;
    $mes = $request->mes;

    if (!$empresaId) {
        return response()->json(['error' => 'Falta filtro de empresa'], 400);
    }

    // Obtener el RUT de la empresa
    $empresa = \App\Models\Parametros\ParEmpresa::find($empresaId);
    if (!$empresa) {
        return response()->json(['error' => 'Empresa no encontrada'], 404);
    }

    $baseQuery = PosMantencionDetalle::whereHas('posMaquinaria.posCliente', function ($q) use ($empresa) {
        $q->where('rut', $empresa->rut);
    });

    // Siempre obtener posMantenciones activas (estado 1) de pos_maquinarias propias
    $mantencionesActivas = $baseQuery->where('estado', 1)
        ->select(
            DB::raw('COUNT(*) as cantidad_mantenciones_activas'),
            DB::raw('COUNT(DISTINCT pos_maquinaria_id) as cantidad_maquinarias_unicas'),
            DB::raw('SUM(valor) as suma_valor_total_activas')
        )
        ->first();

    $resultado = [
        'mantenciones_activas' => $mantencionesActivas
    ];

    // Si hay año y mes, agregar resumen mensual de pos_maquinarias propias
    if ($anio && $mes) {
        $resumenMensual = PosMantencionDetalle::whereHas('posMaquinaria.posCliente', function ($q) use ($empresa) {
            $q->where('rut', $empresa->rut);
        })
        ->whereYear('f_entrada', $anio)
        ->whereMonth('f_entrada', $mes)
        ->where('estado', 0)
        ->select(
            DB::raw('MONTH(f_entrada) as mes'),
            DB::raw('COUNT(*) as cantidad_mantenciones'),
            DB::raw('SUM(valor) as suma_valor_total'),
            DB::raw('ROUND(AVG(valor), 2) as promedio_valor_mantencion')
        )
        ->groupBy(DB::raw('MONTH(f_entrada)'))
        ->orderBy(DB::raw('MONTH(f_entrada)'))
        ->first();

        $resultado['resumen_mensual'] = $resumenMensual;
    }

    return response()->json($resultado);
}

public function resumenMensualATmaquinarias(Request $request)
{
    $empresaId = $request->empresa;
    $anio = $request->anio;

    if (!$empresaId || !$anio) {
        return response()->json(['error' => 'Faltan filtros necesarios'], 400);
    }

    $resumen = PosArriendoDetalle::query()
        ->whereYear('f_salida', $anio)
        ->whereHas('arriendo', function ($q) use ($empresaId) {
            $q->where('par_empresa_id', $empresaId);
        })
    ->join('pos_maquinarias', 'pos_arriendo_detalles.pos_maquinaria_id', '=', 'pos_maquinarias.id')
        ->join('t_maquinarias', 'pos_maquinarias.t_maquinaria_id', '=', 't_maquinarias.id')
        ->select(
            DB::raw('MONTH(f_salida) as mes'),
            't_maquinarias.nombre as tipo_maquinaria',
            DB::raw('SUM(v_total) as total')
        )
        ->groupBy(DB::raw('MONTH(f_salida)'), 't_maquinarias.nombre')
        ->orderBy(DB::raw('MONTH(f_salida)'))
        ->get();

    return response()->json($resumen);
}

public function resumenMensualACTmaquinarias(Request $request)
{
    $empresaId = $request->empresa;
    $anio = $request->anio;

    if (!$empresaId || !$anio) {
        return response()->json(['error' => 'Faltan filtros necesarios'], 400);
    }

    $resumen = PosArriendoDetalle::query()
        ->whereYear('f_salida', $anio)
        ->whereHas('arriendo', function ($q) use ($empresaId) {
            $q->where('par_empresa_id', $empresaId);
        })
    ->join('pos_maquinarias', 'pos_arriendo_detalles.pos_maquinaria_id', '=', 'pos_maquinarias.id')
        ->join('t_maquinarias', 'pos_maquinarias.t_maquinaria_id', '=', 't_maquinarias.id')
        ->select(
            DB::raw('MONTH(f_salida) as mes'),
            't_maquinarias.nombre as tipo_maquinaria',
            DB::raw('SUM(cantidad) as total')
        )
        ->groupBy(DB::raw('MONTH(f_salida)'), 't_maquinarias.nombre')
        ->orderBy(DB::raw('MONTH(f_salida)'))
        ->get();

    return response()->json($resumen);
}

public function resumenMensualMantencion(Request $request)
{
    $empresaId = $request->empresa;
    $anio = $request->anio;

    if (!$empresaId || !$anio) {
        return response()->json(['error' => 'Faltan filtros necesarios'], 400);
    }

  $resumen = PosMDespacho::whereHas('posMantenciones', function ($q) use ($empresaId) {
        $q->where('par_empresa_id', $empresaId);
    })
    ->whereYear('fecha', $anio)
    ->select(
        DB::raw('MONTH(fecha) as mes'),
        DB::raw('SUM(total) as total')
    )
    ->groupBy(DB::raw('MONTH(fecha)'))
    ->orderBy(DB::raw('MONTH(fecha)'))
    ->get();


    // 🔁 Devuelve directamente el número del mes y total
    return response()->json($resumen);
}
public function resumenMensualMantencionTMaquinarias(Request $request)
{
    $empresaId = $request->empresa;
    $anio = $request->anio;

    if (!$empresaId || !$anio) {
        return response()->json(['error' => 'Faltan filtros necesarios'], 400);
    }

    $resumen = PosMDespacho::with('posMantenciones.posMaquinaria.parTMaquinaria')
        ->whereYear('fecha', $anio)
        ->where('par_empresa_id', $empresaId) // 👈 ahora filtra por m_despachos.par_empresa_id
        ->get()
        ->flatMap(function ($despacho) {
            return $despacho->posMantenciones->map(function ($detalle) use ($despacho) {
                return [
                    'mes' => (int) date('n', strtotime($despacho->fecha)),
                    'tipo_maquinaria' => optional($detalle->posMaquinaria->parTMaquinaria)->nombre ?? 'Sin Tipo',
                    'total' => (int) $despacho->total,
                ];
            });
        })
        ->groupBy(fn ($item) => $item['mes'].'-'.$item['tipo_maquinaria'])
        ->map(function ($items) {
            return [
                'mes' => $items->first()['mes'],
                'tipo_maquinaria' => $items->first()['tipo_maquinaria'],
                'total' => $items->sum('total'),
            ];
        })
        ->values();

    return response()->json($resumen);
}


public function resumenMensualMantencionCTMaquinarias(Request $request)
{
    $empresaId = $request->empresa;
    $anio = $request->anio;

    if (!$empresaId || !$anio) {
        return response()->json(['error' => 'Faltan filtros necesarios'], 400);
    }

    // Carga los despachos con posMantenciones y pos_maquinarias
    $resumen = PosMDespacho::with('posMantenciones.posMaquinaria.parTMaquinaria')
        ->whereYear('fecha', $anio)
        ->where('par_empresa_id', $empresaId)
        ->get()
        ->flatMap(function ($despacho) {
            // Agrupamos por posMaquinaria única dentro de un despacho
            return $despacho->posMantenciones
                ->unique('pos_maquinaria_id')
                ->map(function ($detalle) use ($despacho) {
                    return [
                        'mes' => (int) date('n', strtotime($despacho->fecha)),
                        'tipo_maquinaria' => optional($detalle->posMaquinaria->parTMaquinaria)->nombre ?? 'Sin Tipo',
                    ];
                });
        })
        ->groupBy(fn ($item) => $item['mes'].'-'.$item['tipo_maquinaria'])
        ->map(function ($items) {
            return [
                'mes' => $items->first()['mes'],
                'tipo_maquinaria' => $items->first()['tipo_maquinaria'],
                'total' => $items->count(), // 👈 cantidad de posMantenciones únicas por tipo
            ];
        })
        ->values();

    return response()->json($resumen);
}



}
