<?php

namespace App\Http\Controllers;

use App\Models\Contabilidad\ConSiiCompra;
use App\Models\Contabilidad\ConSiiVenta;
use App\Models\Contabilidad\Trabajador;
use App\Models\Contabilidad\Facturacione;
use App\Models\Contabilidad\Maquinaria;
use Carbon\Carbon;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
 public function Impuesto(Request $request)
{
    $filtro = $request->filtro;
    $periodo = $request->periodo;

    // 🟢 COMPRAS
    $compras = DB::table('con_sii_compras')
        ->selectRaw('
            par_empresa_id,
            par_periodo_id,
            SUM(CASE WHEN tipo_doc != 61 THEN 
                COALESCE(Monto_IVA_Recuperable, 0) + 
                COALESCE(Monto_IVA_Uso_Comun, 0) + 
                COALESCE(IVA_Activo_Fijo, 0)
            ELSE 0 END) 
            -
            SUM(CASE WHEN tipo_doc = 61 THEN 
                COALESCE(Monto_IVA_Recuperable, 0) + 
                COALESCE(Monto_IVA_Uso_Comun, 0) + 
                COALESCE(IVA_Activo_Fijo, 0)
            ELSE 0 END) AS total_iva_compras
        ')
        ->where('par_empresa_id', $filtro)
        ->where('par_periodo_id', $periodo)
        ->groupBy('par_empresa_id', 'par_periodo_id');

    // 🔵 VENTAS
    $ventas = DB::table('con_sii_ventas')
        ->selectRaw('
            par_empresa_id,
            par_periodo_id,
            SUM(
                CASE WHEN tipo_doc != 61 THEN 
                    COALESCE(Monto_IVA, 0) + 
                    COALESCE(IVA_propio, 0) + 
                    COALESCE(IVA_Comision_Liquid_Factura, 0) + 
                    COALESCE(IVA_fuera_de_plazo, 0) + 
                    COALESCE(IVA_Retenido_Total, 0) + 
                    COALESCE(IVA_Retenido_Parcial, 0) + 
                    COALESCE(IVA_Terceros, 0)
                ELSE 0
                END
            ) -
            SUM(
                CASE WHEN tipo_doc = 61 THEN 
                    COALESCE(Monto_IVA, 0) + 
                    COALESCE(IVA_propio, 0) + 
                    COALESCE(IVA_Comision_Liquid_Factura, 0) + 
                    COALESCE(IVA_fuera_de_plazo, 0) + 
                    COALESCE(IVA_Retenido_Total, 0) + 
                    COALESCE(IVA_Retenido_Parcial, 0) + 
                    COALESCE(IVA_Terceros, 0)
                ELSE 0
                END
            ) AS total_iva_ventas
        ')
        ->where('par_empresa_id', $filtro)
        ->where('par_periodo_id', $periodo)
        ->groupBy('par_empresa_id', 'par_periodo_id');

    // 🔸 PPM
    $ppm = DB::table('par_empresas')
        ->where('id', $filtro)
        ->value('ppm');

    // 🔸 Ventas netas sin tipo_doc 61
    $ventasNeto = DB::table('con_sii_ventas')
        ->where('par_empresa_id', $filtro)
        ->where('par_periodo_id', $periodo)
        ->where('tipo_doc', '!=', 61)
        ->selectRaw('
            SUM(
                COALESCE(Monto_Neto, 0) + 
                COALESCE(Monto_Exento, 0) + 
                COALESCE(Neto_Comision_Liquid_Factura, 0) + 
                COALESCE(Exento_Comision_Liquid_Factura, 0) + 
                COALESCE(Venta_Pasajes_Transporte_Nacional, 0) + 
                COALESCE(Venta_Pasajes_Transporte_Internacional, 0)
            ) as total
        ')
        ->value('total') ?? 0;

    // 🔸 Ventas con tipo_doc = 61
    $ventasCredito = DB::table('con_sii_ventas')
        ->where('par_empresa_id', $filtro)
        ->where('par_periodo_id', $periodo)
        ->where('tipo_doc', 61)
        ->selectRaw('
            SUM(
                COALESCE(Monto_Neto, 0) + 
                COALESCE(Monto_Exento, 0) + 
                COALESCE(Neto_Comision_Liquid_Factura, 0) + 
                COALESCE(Exento_Comision_Liquid_Factura, 0) + 
                COALESCE(Venta_Pasajes_Transporte_Nacional, 0) + 
                COALESCE(Venta_Pasajes_Transporte_Internacional, 0)
            ) as total
        ')
        ->value('total') ?? 0;

    // Obtener códigos de impuestos que pertenecen al grupo ANTICIPADO
    $codigosAnticipados = DB::table('con_impuestos')
        ->where('grupo', 'ANTICIPADO')
        ->where('estado', 1)
        ->pluck('codigo')
        ->toArray();

    $impuestoAnticipado = 0;
    
    if (!empty($codigosAnticipados)) {
        $impuestoAnticipado = DB::table('con_sii_compras')
            ->where('par_empresa_id', $filtro)
            ->where('par_periodo_id', $periodo)
            ->whereIn('Codigo_Otro_Impuesto', $codigosAnticipados)
            ->selectRaw('
                SUM(CASE WHEN tipo_doc != 61 THEN 
                    COALESCE(Valor_Otro_Impuesto, 0)
                ELSE 0 END) 
                -
                SUM(CASE WHEN tipo_doc = 61 THEN 
                    COALESCE(Valor_Otro_Impuesto, 0)
                ELSE 0 END) as total
            ')
            ->value('total') ?? 0;
    }

    // 🔸 Base PPM
    $ppmDecimal = floatval($ppm);
    $basePPM = ($ventasNeto - $ventasCredito) * $ppmDecimal;

    // 🔸 Remanente par_mese anterior (código 504)
    $remanente = DB::table('con_formulario_detalles as d')
    ->join('con_par_formularios as p', 'd.con_par_formulario_id', '=', 'p.id')
    ->where('d.par_empresa_id', $filtro)
    ->where('d.par_periodo_id', $periodo)
    ->where('p.codigo', 504)
    ->value('d.monto') ?? 0;

    // 🟡 Resultado IVA
    $resultado = DB::table(DB::raw("({$compras->toSql()}) as compras"))
        ->mergeBindings($compras)
        ->leftJoinSub($ventas, 'ventas', function ($join) {
            $join->on('compras.par_empresa_id', '=', 'ventas.par_empresa_id')
                 ->on('compras.par_periodo_id', '=', 'ventas.par_periodo_id');
        })
        ->select(
            'compras.par_empresa_id',
            'compras.par_periodo_id',
            DB::raw('COALESCE(compras.total_iva_compras, 0) as total_iva_compras'),
            DB::raw('COALESCE(ventas.total_iva_ventas, 0) as total_iva_ventas'),
            DB::raw('COALESCE(ventas.total_iva_ventas, 0) - COALESCE(compras.total_iva_compras, 0) as diferencia_iva')
        )
        ->first();

    if (!$resultado) {
        $resultado = (object)[
            'par_empresa_id' => $filtro,
            'par_periodo_id' => $periodo,
            'total_iva_compras' => 0,
            'total_iva_ventas' => 0,
            'diferencia_iva' => 0,
        ];
    }

$ivaMasRemanente = 0;

if ($resultado->diferencia_iva > 0) {
    $ivaMasRemanente = $resultado->diferencia_iva - $remanente + $basePPM - $impuestoAnticipado;
} elseif (!$basePPM || $basePPM == 0) {
    if ($remanente > 0) {
        $ivaMasRemanente = $remanente - $impuestoAnticipado;
    } else {
        $ivaMasRemanente = abs($resultado->diferencia_iva) - $impuestoAnticipado;
    }
} else {
    $ivaMasRemanente = $basePPM - $impuestoAnticipado;
}

// 🏷️ Etiqueta para mostrar en frontend
$etiquetaIvaFinal = '';

if ($resultado->diferencia_iva > 0) {
    $etiquetaIvaFinal = 'Impuesto a Pagar + PPM - Remanente';
} elseif (!$basePPM || $basePPM == 0) {
    if ($remanente > 0) {
        $etiquetaIvaFinal = 'Remanente Mes Anterior';
    } else {
        $etiquetaIvaFinal = 'Diferencia de IVA Positiva';
    }
} else {
    $etiquetaIvaFinal = 'PPM del Mes';
}

// 👇 Obtenemos los valores necesarios
$ventasNetoReal = $ventasNeto - $ventasCredito;
$ivaVentas = $resultado->total_iva_ventas;
$totalVentas = $ventasNetoReal + $ivaVentas;

$ivaCompras = $resultado->total_iva_compras;
$comprasNeto = DB::table('con_sii_compras')
    ->where('par_empresa_id', $filtro)
    ->where('par_periodo_id', $periodo)
    ->selectRaw('
        SUM(CASE WHEN tipo_doc != 61 THEN 
            COALESCE(Monto_Neto, 0) + COALESCE(Monto_Exento, 0)
        ELSE 0 END) 
        -
        SUM(CASE WHEN tipo_doc = 61 THEN 
            COALESCE(Monto_Neto, 0) + COALESCE(Monto_Exento, 0)
        ELSE 0 END) as total
    ')
    ->value('total') ?? 0;
$totalCompras = $comprasNeto + $ivaCompras;

$utilidadBruta = $totalVentas - $totalCompras;
$margenEstimado = $totalVentas > 0 ? round(($utilidadBruta / $totalVentas) * 100, 1) : 0;




// 📅 Obtener el período actual
$periodoActual = DB::table('par_periodos')
    ->where('id', $periodo)
    ->first();

$anioActual = DB::table('par_anios')
    ->where('id', $periodoActual->par_anio_id)
    ->value('nro');

$mesActual = DB::table('par_meses')
    ->where('id', $periodoActual->par_mese_id ?? $periodoActual->par_mese_id)
    ->value('nro');

// 📆 Generar últimos 12 periodos
$ultimosPeriodos = [];
for ($i = 0; $i < 12; $i++) {
    $fecha = Carbon::createFromDate($anioActual, $mesActual, 1)->subMonths($i);

    $anioId = DB::table('par_anios')->where('nro', $fecha->year)->value('id');
    $mesId = DB::table('par_meses')->where('nro', $fecha->month)->value('id');

    if ($anioId && $mesId) {
        $periodoId = DB::table('par_periodos')
            ->where('par_anio_id', $anioId)
            ->where(function($q) use ($mesId) {
                                $q->where('par_mese_id', $mesId)
                                    ->orWhere('par_mese_id', $mesId);
            })
            ->value('id');

        if ($periodoId) {
            $label = $fecha->format('M-Y'); // Ej: "Aug-2025"
            $ultimosPeriodos[] = [
                'label' => $label,
                'par_periodo_id' => $periodoId,
            ];
        }
    }
}

// 🔢 Totales compras y ventas mensualizadas
$comprasMensuales = DB::table('con_sii_compras')
    ->select('par_periodo_id', DB::raw('SUM(Monto_Neto + Monto_Exento) as neto'), DB::raw('SUM(COALESCE(Monto_IVA_Recuperable, 0) + COALESCE(Monto_IVA_Uso_Comun, 0) + COALESCE(IVA_Activo_Fijo, 0)) as iva'))
    ->where('par_empresa_id', $filtro)
    ->whereIn('par_periodo_id', array_column($ultimosPeriodos, 'par_periodo_id'))
    ->groupBy('par_periodo_id')
    ->get()
    ->keyBy('par_periodo_id');

$ventasMensuales = DB::table('con_sii_ventas')
    ->select('par_periodo_id', DB::raw('SUM(Monto_Neto + Monto_Exento) as neto'), DB::raw('SUM(COALESCE(Monto_IVA, 0) + COALESCE(IVA_propio, 0) + COALESCE(IVA_Comision_Liquid_Factura, 0) + COALESCE(IVA_fuera_de_plazo, 0) + COALESCE(IVA_Retenido_Total, 0) + COALESCE(IVA_Retenido_Parcial, 0) + COALESCE(IVA_Terceros, 0)) as iva'))
    ->where('par_empresa_id', $filtro)
    ->whereIn('par_periodo_id', array_column($ultimosPeriodos, 'par_periodo_id'))
    ->groupBy('par_periodo_id')
    ->get()
    ->keyBy('par_periodo_id');

$mensualizados = [];

foreach (array_reverse($ultimosPeriodos) as $item) {
    $pid = $item['par_periodo_id'];
    $label = $item['label'];

    $venta = $ventasMensuales[$pid] ?? (object)['neto' => 0, 'iva' => 0];
    $compra = $comprasMensuales[$pid] ?? (object)['neto' => 0, 'iva' => 0];

    $mensualizados[] = [
        'label' => $label,
        'ventas_total' => floatval($venta->neto + $venta->iva),
        'compras_total' => floatval($compra->neto + $compra->iva),
    ];
}

$ventasPorCliente = DB::table('con_sii_ventas as v')
    ->join('pos_clientes as c', 'v.pos_cliente_id', '=', 'c.id')
    ->select(
       'v.pos_cliente_id as cliente_id',
        'c.nombre as cliente_nombre',
        'v.par_periodo_id',

        // Neto con notas de crédito restadas
        DB::raw('SUM(
            CASE 
                WHEN v.tipo_doc = 61 THEN -(v.Monto_Neto + v.Monto_Exento) 
                ELSE (v.Monto_Neto + v.Monto_Exento) 
            END
        ) as neto'),

        // IVA con notas de crédito restadas
        DB::raw('SUM(
            CASE 
                WHEN v.tipo_doc = 61 THEN -(
                    COALESCE(v.Monto_IVA, 0) +
                    COALESCE(v.IVA_propio, 0) +
                    COALESCE(v.IVA_Comision_Liquid_Factura, 0) +
                    COALESCE(v.IVA_fuera_de_plazo, 0) +
                    COALESCE(v.IVA_Retenido_Total, 0) +
                    COALESCE(v.IVA_Retenido_Parcial, 0) +
                    COALESCE(v.IVA_Terceros, 0)
                )
                ELSE (
                    COALESCE(v.Monto_IVA, 0) +
                    COALESCE(v.IVA_propio, 0) +
                    COALESCE(v.IVA_Comision_Liquid_Factura, 0) +
                    COALESCE(v.IVA_fuera_de_plazo, 0) +
                    COALESCE(v.IVA_Retenido_Total, 0) +
                    COALESCE(v.IVA_Retenido_Parcial, 0) +
                    COALESCE(v.IVA_Terceros, 0)
                )
            END
        ) as iva')
    )
    ->where('v.par_empresa_id', $filtro)
    ->whereIn('v.par_periodo_id', array_column($ultimosPeriodos, 'par_periodo_id'))
    ->groupBy('v.pos_cliente_id', 'v.par_periodo_id', 'c.nombre')
    ->get();




  $ventasPorClienteMensual = [];

foreach ($ventasPorCliente as $v) {
    $pid = $v->par_periodo_id;
    $cid = $v->cliente_id;

    if (!isset($ventasPorClienteMensual[$pid])) {
        $ventasPorClienteMensual[$pid] = [];
    }

    $ventasPorClienteMensual[$pid][] = [
        'cliente_id' => $cid,
        'cliente_nombre' => $v->cliente_nombre, // 👈 ahora sí disponible
        'neto' => floatval($v->neto),
        'iva' => floatval($v->iva),
        'total' => floatval($v->neto + $v->iva),
    ];
}

$ventasPorClienteTotal = [];

foreach ($ventasPorCliente as $v) {
    $cid = $v->cliente_id;

    if (!isset($ventasPorClienteTotal[$cid])) {
        $ventasPorClienteTotal[$cid] = [
            'cliente_id' => $cid,
            'cliente_nombre' => $v->cliente_nombre,
            'neto' => 0,
            'iva' => 0,
            'total' => 0
        ];
    }

    $ventasPorClienteTotal[$cid]['neto'] += floatval($v->neto);
    $ventasPorClienteTotal[$cid]['iva'] += floatval($v->iva);
    $ventasPorClienteTotal[$cid]['total'] += floatval($v->neto + $v->iva);
}

// Si deseas que sea un array indexado:
$ventasPorClienteTotal = array_values($ventasPorClienteTotal);


    // 🔚 Respuesta final
return response()->json([
    'iva' => $resultado,
    'ventas_netas' => $ventasNetoReal,
    'ppm' => $ppm,
    'ppm_calculado' => $basePPM,
    'remanente_mes_anterior' => $remanente,
    'iva_mas_remanente' => $ivaMasRemanente,
    'etiqueta_iva_final' => $etiquetaIvaFinal,
    'impuesto_anticipado' => $impuestoAnticipado,

    // 🧾 Tarjeta Ventas
    'resumen_ventas' => [
        'neto' => $ventasNetoReal,
        'iva' => $ivaVentas,
        'total' => $totalVentas
    ],

    // 🛒 Tarjeta Compras
    'resumen_compras' => [
        'neto' => $comprasNeto,
        'iva' => $ivaCompras,
        'total' => $totalCompras
    ],

    // 💰 Tarjeta Utilidad
    'resumen_utilidad' => [
        'utilidad_bruta' => $utilidadBruta,
        'margen' => $margenEstimado,
        'proyeccion' => $utilidadBruta
    ],

    'ventas_compras_mensual' => $mensualizados,
    'ventas_por_cliente_mensual' => $ventasPorClienteMensual,
    'ventas_por_cliente_total' => $ventasPorClienteTotal,



]);







}

public function ImpuestoGraficoDesdePeriodo(Request $request)
{
    $empresaId = $request->filtro;
    $periodoId = $request->periodo;

    $periodoActual = DB::table('par_periodos')->where('id', $periodoId)->first();

    if (!$periodoActual) {
        return response()->json(['error' => 'Periodo no encontrado'], 404);
    }

    // Obtener los 12 periodos hacia atrás desde el seleccionado
    $ultimosPeriodos = DB::table('par_periodos')
        ->where(function ($query) use ($periodoActual) {
            $query->where('anio', '<', $periodoActual->anio)
                  ->orWhere(function ($q) use ($periodoActual) {
                      $q->where('anio', '=', $periodoActual->anio)
                        ->where('par_mese', '<=', $periodoActual->par_mese);
                  });
        })
        ->orderByDesc('anio')
        ->orderByDesc('par_mese')
        ->limit(12)
        ->get();

    $datos = [];

    foreach ($ultimosPeriodos as $p) {
        $periodoId = $p->id;
        $periodoLabel = $p->anio . '-' . str_pad($p->par_mese, 2, '0', STR_PAD_LEFT);

        // 🔵 VENTAS
        $ivaVentas = DB::table('con_sii_ventas')
            ->where('par_empresa_id', $empresaId)
            ->where('par_periodo_id', $periodoId)
            ->selectRaw("
                SUM(CASE WHEN tipo_doc != 61 THEN 
                    COALESCE(Monto_IVA, 0) + COALESCE(IVA_propio, 0) +
                    COALESCE(IVA_Comision_Liquid_Factura, 0) + COALESCE(IVA_fuera_de_plazo, 0) +
                    COALESCE(IVA_Retenido_Total, 0) + COALESCE(IVA_Retenido_Parcial, 0) +
                    COALESCE(IVA_Terceros, 0)
                ELSE 0 END)
                -
                SUM(CASE WHEN tipo_doc = 61 THEN 
                    COALESCE(Monto_IVA, 0) + COALESCE(IVA_propio, 0) +
                    COALESCE(IVA_Comision_Liquid_Factura, 0) + COALESCE(IVA_fuera_de_plazo, 0) +
                    COALESCE(IVA_Retenido_Total, 0) + COALESCE(IVA_Retenido_Parcial, 0) +
                    COALESCE(IVA_Terceros, 0)
                ELSE 0 END) as total
            ")
            ->value('total') ?? 0;

        // 🟢 COMPRAS
        $ivaCompras = DB::table('con_sii_compras')
            ->where('par_empresa_id', $empresaId)
            ->where('par_periodo_id', $periodoId)
            ->selectRaw("
                SUM(CASE WHEN tipo_doc != 61 THEN 
                    COALESCE(Monto_IVA_Recuperable, 0) + COALESCE(Monto_IVA_Uso_Comun, 0) + 
                    COALESCE(IVA_Activo_Fijo, 0)
                ELSE 0 END)
                -
                SUM(CASE WHEN tipo_doc = 61 THEN 
                    COALESCE(Monto_IVA_Recuperable, 0) + COALESCE(Monto_IVA_Uso_Comun, 0) + 
                    COALESCE(IVA_Activo_Fijo, 0)
                ELSE 0 END) as total
            ")
            ->value('total') ?? 0;

        $datos[] = [
            'periodo' => $periodoLabel,
            'ventas' => round($ivaVentas),
            'compras' => round($ivaCompras),
        ];
    }

    // Ordenar por fecha ASC
    $datos = array_reverse($datos);

    return response()->json($datos);
}





}
