<?php

namespace App\Http\Controllers;

use App\Libraries\FirmaElectronica;
use App\Models\Venta;
use App\Models\Caf;
use App\Models\Dte;
use App\Models\FolioUsado;
use App\Http\Controllers\DteLogController;
use Illuminate\Http\Request;
use sasco\LibreDTE\FirmaElectronica as LibreFirma;
use sasco\LibreDTE\DTE as LibreDTE;
use sasco\LibreDTE\Certificacion;
use sasco\LibreDTE\XML;

class DteController extends Controller
{
    public function generarDTE(\$venta_id)
    {
        \$venta = Venta::with(['cliente', 'empresa', 'venta_inventarios'])->findOrFail(\$venta_id);

        if (!\$venta->folio) {
            return response()->json(['error' => 'La venta no tiene folio asignado.'], 400);
        }

        if (!\$venta->cliente_id) {
            return response()->json(['error' => 'La venta no tiene un cliente asignado.'], 400);
        }

        \$caf = Caf::where('empresa_id', \$venta->empresa_id)
            ->where('tipo_dte', 33)
            ->where('folio_desde', '<=', \$venta->folio)
            ->where('folio_hasta', '>=', \$venta->folio)
            ->first();

        if (!\$caf) {
            return response()->json(['error' => 'No se encontró un CAF que contenga el folio de esta venta.'], 400);
        }

        // Firma
        \$certPath = storage_path('app' . env('CERTIFICADO_PFX'));
        \$firma = new LibreFirma(file_get_contents(\$certPath), env('CERTIFICADO_PASSWORD'));

        \$datos = [
            'Encabezado' => [
                'IdDoc' => [
                    'TipoDTE' => 33,
                    'Folio' => \$venta->folio,
                    'FchEmis' => \$venta->fecha,
                ],
                'Emisor' => [
                    'RUTEmisor' => \$venta->empresa->rut,
                    'RznSoc' => \$venta->empresa->razon_social,
                    'GiroEmis' => \$venta->empresa->giro,
                    'Acteco' => '671090',
                ],
                'Receptor' => [
                    'RUTRecep' => \$venta->cliente->rut,
                    'RznSocRecep' => \$venta->cliente->nombre,
                    'GiroRecep' => \$venta->cliente->giro,
                    'DirRecep' => \$venta->cliente->direccion,
                ],
                'Totales' => [
                    'MntNeto' => round(\$venta->neto),
                    'IVA' => round(\$venta->impuesto),
                    'MntTotal' => round(\$venta->total),
                ],
            ],
        ];

        foreach (\$venta->venta_inventarios as \$i => \$item) {
            \$datos['Detalle'][] = [
                'NroLinDet' => \$i + 1,
                'NmbItem' => \$item->texto,
                'QtyItem' => number_format(\$item->cantidad, 2, '.', ''),
                'PrcItem' => round(\$item->neto / max(\$item->cantidad, 1)),
                'MontoItem' => round(\$item->neto),
                'UnmdItem' => 'UNI',
            ];
        }

        \$dte = new LibreDTE([
            'Encabezado' => \$datos['Encabezado'],
            'Detalle' => \$datos['Detalle'],
        ]);

        \$dte->timbrar(\$caf->xml);
        \$dte->firmar(\$firma);
        \$xml = \$dte->generar();

        \$rutaDirectorio = storage_path("app/dtes/empresa_{\$venta->empresa_id}/");
        if (!file_exists(\$rutaDirectorio)) {
            mkdir(\$rutaDirectorio, 0777, true);
        }

        \$nombreArchivo = "DTE_33_F{\$venta->folio}.xml";
        \$rutaCompleta = \$rutaDirectorio . \$nombreArchivo;
        file_put_contents(\$rutaCompleta, \$xml);

        Dte::updateOrCreate(
            ['venta_id' => \$venta->id],
            [
                'empresa_id' => \$venta->empresa_id,
                'tipo_dte' => 33,
                'cliente_id' => \$venta->cliente_id,
                'fecha_emision' => \$venta->fecha,
                'folio' => \$venta->folio,
                'estado' => 'GENERADO',
                'ruta_xml' => "dtes/empresa_{\$venta->empresa_id}/{\$nombreArchivo}",
                'xml' => "dtes/empresa_{\$venta->empresa_id}/{\$nombreArchivo}",
                'fecha' => now(),
                'neto' => \$venta->neto,
                'iva' => \$venta->impuesto,
                'exento' => \$venta->exento ?? 0,
                'total' => \$venta->total
            ]
        );

        FolioUsado::updateOrCreate(
            [
                'empresa_id' => \$venta->empresa_id,
                'tipo_dte' => 33,
                'folio' => \$venta->folio
            ],
            [
                'fecha' => now(),
                'venta_id' => \$venta->id
            ]
        );

        DteLogController::storeLog(
            \$venta->id,
            "XML generado con TED y guardado en {\$rutaCompleta}.",
            \$venta->cliente_id
        );

        return response()->json([
            'mensaje' => 'XML con TED generado correctamente.',
            'ruta' => \$rutaCompleta
        ]);
    }
}

