<?php

namespace App\Http\Controllers\Pos;

use App\Models\Parametros\ParSucursal as ParSucursal;
use App\Models\Pos\PosBodega;
use App\Models\Pos\PosArticulo;
use Illuminate\Http\Request;
use App\Models\Pos\PosBodegaArticulo;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use App\Models\Pos\PosArticuloImpuesto;
use App\Models\Pos\PosArticuloPackItem;
use App\Models\Pos\PosBodegaArticuloImpuesto;
use App\Models\Pos\PosBodegaArticuloPackItem;

class PosArticuloController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return PosArticulo::with(['PosBodega','Categoria','Medida'])->where('estado',1)->get();
       

    }

public function filtroEmpresa(Request $request)
{
    $empresaId = $request->input('empresa');


    return PosArticulo::with(['PosBodega','Categoria','Medida'])
    ->where('estado',1)
    ->where('par_empresa_id', $empresaId)

        ->get();
}

public function filtroBAE(Request $request)
{
    $empresaId = $request->input('empresa');

    // ✅ Traer datos desde pos_bodega_articulos con información del artículo
    $articulos = PosBodegaArticulo::join('pos_articulos', 'pos_bodega_articulos.pos_articulo_id', '=', 'pos_articulos.id')
        ->leftJoin('pos_bodegas', 'pos_bodega_articulos.pos_bodega_id', '=', 'pos_bodegas.id')
        ->leftJoin('par_medidas', 'pos_articulos.par_medida_id', '=', 'par_medidas.id')
        ->leftJoin('par_marcas', 'pos_articulos.par_marca_id', '=', 'par_marcas.id')
        ->leftJoin('pos_categorias', 'pos_articulos.pos_categoria_id', '=', 'pos_categorias.id')
    ->where('pos_bodega_articulos.par_empresa_id', $empresaId)
        ->where('pos_articulos.estado', 1)
        ->select(
            'pos_bodega_articulos.*',
            'pos_bodega_articulos.p_venta as venta',
            'pos_bodega_articulos.pos_bodega_id as bodega_id',
            'pos_articulos.nombre as nombre',
            'pos_articulos.barra',
            'pos_articulos.par_marca_id',
            'pos_articulos.par_medida_id',
            'pos_articulos.pos_categoria_id',
            'pos_bodegas.nombre as nombre_bodega',
            'par_medidas.nombre as nombre_medida',
            'par_marcas.nombre as nombre_marca',
            'pos_categorias.nombre as nombre_categoria'
        )
        ->orderBy('pos_bodegas.nombre', 'asc')
        ->orderBy('pos_articulos.nombre', 'asc')
        ->get();

    // Cargar impuestos para cada BA desde pos_bodega_articulos_impuestos
    $pbaIds = $articulos->pluck('id')->filter()->unique()->values()->all();
    if (!empty($pbaIds)) {
        $impuestosPorPba = collect(
            DB::table('pos_bodega_articulo_impuestos as pbai')
                ->join('con_impuestos as pi', 'pbai.con_impuesto_id', '=', 'pi.id')
                ->whereIn('pbai.pos_bodega_articulo_id', $pbaIds)
                ->select('pbai.pos_bodega_articulo_id as pba_id', 'pbai.con_impuesto_id', 'pbai.activo', 'pbai.optativo_en_venta', 'pi.codigo', 'pi.nombre')
                ->get()
        )->groupBy('pba_id')->map(function ($g) {
            return $g->map(function ($r) {
                return [
                    'con_impuesto_id'   => (int) $r->con_impuesto_id,
                    'activo'            => (int) $r->activo,
                    'optativo_en_venta' => (int) $r->optativo_en_venta,
                    'codigo'            => $r->codigo,
                    'nombre'            => $r->nombre,
                ];
            })->values()->all();
        });

        // Anexar a cada artículo la lista de impuestos
        $articulos = $articulos->map(function ($a) use ($impuestosPorPba) {
            $a->impuestos = $impuestosPorPba->get($a->id, []);
            return $a;
        });
    }

    return $articulos;
}

public function filtroESBodega(Request $request)
{
    $empresaId = $request->input('empresa');
    $sucursalId = $request->input('sucursal');
    $bodegaId = $request->input('bodega');

    // ✅ Traer datos desde pos_bodega_articulos con información del artículo
    $articulos = PosBodegaArticulo::join('pos_articulos', 'pos_bodega_articulos.pos_articulo_id', '=', 'pos_articulos.id')
        ->leftJoin('pos_bodegas', 'pos_bodega_articulos.pos_bodega_id', '=', 'pos_bodegas.id')
        ->leftJoin('par_medidas', 'pos_articulos.par_medida_id', '=', 'par_medidas.id')
        ->leftJoin('par_marcas', 'pos_articulos.par_marca_id', '=', 'par_marcas.id')
        ->leftJoin('pos_categorias', 'pos_articulos.pos_categoria_id', '=', 'pos_categorias.id')
    ->where('pos_bodega_articulos.par_empresa_id', $empresaId)
    ->where('pos_bodega_articulos.par_sucursal_id', $sucursalId)
    ->where('pos_bodega_articulos.pos_bodega_id', $bodegaId)
        ->where('pos_articulos.estado', 1)
        ->select(
            'pos_bodega_articulos.*',
            'pos_bodega_articulos.p_venta as venta',
            'pos_bodega_articulos.pos_bodega_id as bodega_id',
            'pos_articulos.nombre as nombre',
            'pos_articulos.barra',
            'pos_articulos.par_marca_id',
            'pos_articulos.par_medida_id',
            'pos_articulos.pos_categoria_id',
            'pos_bodegas.nombre as nombre_bodega',
            'par_medidas.nombre as nombre_medida',
            'par_marcas.nombre as nombre_marca',
            'pos_categorias.nombre as nombre_categoria'
        )
        ->orderBy('pos_bodegas.nombre', 'asc')
        ->orderBy('pos_articulos.nombre', 'asc')
        ->get();

        // Cargar impuestos para cada BA desde pos_bodega_articulos_impuestos
        $pbaIds = $articulos->pluck('id')->filter()->unique()->values()->all();
        if (!empty($pbaIds)) {
            // 1) obtenemos las filas de impuestos por BA (con campos BA-level)
            $rows = DB::table('pos_bodega_articulo_impuestos as pbai')
                ->whereIn('pbai.pos_bodega_articulo_id', $pbaIds)
                ->select('pbai.pos_bodega_articulo_id as pba_id', 'pbai.con_impuesto_id', 'pbai.activo', 'pbai.optativo_en_venta')
                ->get();

            $impuestosPorPba = collect($rows)->groupBy('pba_id')->map(function ($g) {
                return $g->map(function ($r) {
                    return [
                        'con_impuesto_id'   => (int) $r->con_impuesto_id,
                        'activo'            => (int) $r->activo,
                        'optativo_en_venta' => (int) $r->optativo_en_venta,
                    ];
                })->values()->all();
            });

            // 2) Traer datos completos de con_impuestos para los con_impuesto_id usados
            $impuestoIds = collect($rows)->pluck('con_impuesto_id')->unique()->values()->all();
            $impuestosFull = [];
            if (!empty($impuestoIds)) {
                $impuestosFull = DB::table('con_impuestos')->whereIn('id', $impuestoIds)->get()->keyBy('id');
            }

            // 3) Anexar la fila completa del impuesto a cada entrada BA-level
            $impuestosPorPba = $impuestosPorPba->map(function ($list) use ($impuestosFull) {
                return array_map(function ($it) use ($impuestosFull) {
                    $full = $impuestosFull[$it['con_impuesto_id']] ?? null;
                    $it['impuesto'] = $full ? (array) $full : null;
                    return $it;
                }, $list);
            });

            // Anexar a cada artículo la lista de impuestos
            $articulos = $articulos->map(function ($a) use ($impuestosPorPba) {
                $a->impuestos = $impuestosPorPba->get($a->id, []);
                return $a;
            });
        }

        return $articulos;
}

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
   
   
public function store(Request $request)
{
    return DB::transaction(function () use ($request) {

        // =========================
        // A) Detectar modo (PACK vs VARIEDADES vs SIMPLE)
        // =========================
        $tipoInput = $request->input('tipo', 'simple'); // simple | variante | pack

        // PACK
        $packItemsInput = collect($request->input('pack_items', []))
            ->filter(fn ($it) => is_array($it) && !empty($it['item_articulo_id']));
        $esPack = $packItemsInput->isNotEmpty();

        // VARIEDADES (frontend)
        $variedadesInput = collect($request->input('variedades', []))
            ->filter(fn ($v) => is_array($v) && (!empty($v['presentacion']) || !empty($v['nombre'])));
        $esVariedades = !$esPack && $tipoInput === 'variante' && $variedadesInput->isNotEmpty();

        // Tipo a persistir en el artículo principal (tu convención: variedades => 'simple')
        $tipoParaDB = $esPack ? 'pack' : 'simple';

        // =========================
        // 1) ARTÍCULO PRINCIPAL
        // =========================
        $posArticulo = new PosArticulo();

        // Ajuste: asegurar la columna correcta de "parent" en null
        if (Schema::hasColumn('pos_articulos', 'parent_articulo_id')) {
            $posArticulo->parent_articulo_id = null;
        }
        if (Schema::hasColumn('pos_articulos', 'parent_id')) {
            $posArticulo->parent_id = null;
        }

        $posArticulo->tipo             = $tipoParaDB;
        $posArticulo->nombre           = $request->nombre;
        $posArticulo->sku              = $request->sku ?: null; // trigger genera si es null/''.
        $posArticulo->barra            = $request->barra ?: null;
        $posArticulo->par_medida_id        = (int) $request->par_medida_id;
        $posArticulo->par_marca_id         = (int) $request->par_marca_id;
        $posArticulo->pos_categoria_id     = (int) $request->pos_categoria_id;
        $posArticulo->venta            = (float) ($request->venta ?? 0);
        $posArticulo->peso             = (float) ($request->peso ?? 0);
        $posArticulo->stock_minimo     = (float) ($request->stock_minimo ?? 0);
        $posArticulo->par_empresa_id       = (int) $request->par_empresa_id;
        $posArticulo->estado           = isset($request->estado) ? (int)$request->estado : 1;
        if (Schema::hasColumn('pos_articulos', 'atributos_json'))  { $posArticulo->atributos_json = null; }
        if (Schema::hasColumn('pos_articulos', 'atributos_hash'))  { $posArticulo->atributos_hash = null; }
        $posArticulo->save();
        $posArticulo->refresh(); // lee SKU generado

        // =========================
        // 2) IMPUESTOS del principal
        // =========================
        $impuestosReq = collect($request->input('impuestos', []))
            ->filter(fn ($i) => is_array($i) && !empty($i['con_impuesto_id']))
            ->unique('con_impuesto_id')
            ->map(fn ($i) => [
                'pos_articulo_id'   => $posArticulo->id,
                'con_impuesto_id'   => (int) $i['con_impuesto_id'],
                'activo'            => (int) ($i['activo'] ?? 1),
                'optativo_en_venta' => (int) ($i['optativo_en_venta'] ?? 0),
                'estado'            => isset($i['estado']) ? (int) $i['estado'] : 1,
                'created_at'        => now(),
                'updated_at'        => now(),
            ])
            ->values()
            ->all();

        if (!empty($impuestosReq)) {
            PosArticuloImpuesto::insert($impuestosReq);
        }

        // =========================
        // 3) BA del principal (todas las bodegas)
        // =========================
    $sucursalesIds = ParSucursal::where('par_empresa_id', $request->par_empresa_id)->pluck('id');
    $bodegas       = PosBodega::whereIn('par_sucursal_id', $sucursalesIds)->get(['id','par_sucursal_id']);

        // Mapa: pos_bodega_id -> id BA padre (para enlazar variedades)
        $baPadrePorBodega = [];

        foreach ($bodegas as $b) {
            $ba = PosBodegaArticulo::create([
                'pos_articulo_id'           => $posArticulo->id,
                'pos_bodega_id'             => $b->id,
                'parent_bodega_articulo_id' => null,
                'stock'                     => 0,
                'c_stock'                   => (int) ($request->c_stock ?? 1),
                'c_unitario'                => 0,
                'p_venta'                   => (float) ($request->venta ?? 0),
                'costo'                     => (float) ($request->costo ?? 0),
                'par_empresa_id'            => (int) $request->par_empresa_id,
                'par_sucursal_id'               => (int) $b->par_sucursal_id,
                'estado'                    => isset($request->estado) ? (int)$request->estado : 1,
                'created_at'                => now(),
                'updated_at'                => now(),
            ]);

            $baPadrePorBodega[$b->id] = $ba->id;

            // Clonar impuestos al BA
            if (!empty($impuestosReq)) {
                $rows = array_map(fn ($i) => [
                    'pos_bodega_articulo_id' => $ba->id,
                    'con_impuesto_id'        => $i['con_impuesto_id'],
                    'activo'                 => $i['activo'],
                    'optativo_en_venta'      => $i['optativo_en_venta'],
                    'created_at'             => now(),
                    'updated_at'             => now(),
                ], $impuestosReq);
                PosBodegaArticuloImpuesto::insert($rows);
            }
        }

        // =========================
        // 4) PACK (exclusivo)
        // =========================
        if ($esPack) {
            $packItems = $packItemsInput
                ->map(fn ($it) => [
                    'item_articulo_id' => (int) $it['item_articulo_id'],
                    'cantidad'         => (float) ($it['cantidad'] ?? 1),
                    'afecta_stock'     => (int) ($it['afecta_stock'] ?? 1),
                ])
                ->values();

            if ($packItems->isNotEmpty()) {
                // Maestro
                $rows = $packItems->map(function ($it) use ($posArticulo) {
                    return [
                        'pack_id'          => $posArticulo->id,
                        'item_articulo_id' => $it['item_articulo_id'],
                        'cantidad'         => $it['cantidad'],
                        'afecta_stock'     => $it['afecta_stock'],
                        'estado'           => 1,
                        'created_at'       => now(),
                        'updated_at'       => now(),
                    ];
                })->all();
                PosArticuloPackItem::insert($rows);

                // Por bodega: asegurar BA de cada item y relación pba_pack_items
                foreach ($bodegas as $b) {
                    $pbaPackId = $baPadrePorBodega[$b->id] ?? null;
                    if (!$pbaPackId) continue;

                    foreach ($packItems as $it) {
                        $pbaItem = PosBodegaArticulo::firstOrCreate(
                            [
                                'pos_articulo_id' => $it['item_articulo_id'],
                                'pos_bodega_id'   => $b->id,
                            ],
                            [
                                'estado'       => (int) ($request->input('estado', 1)),
                                'stock'        => 0,
                                'c_stock'      => 1,
                                'c_unitario'   => 0,
                                'p_venta'      => 0,
                                'costo'        => 0,
                                'par_empresa_id'   => (int) $request->par_empresa_id,
                                'par_sucursal_id'  => (int) $b->par_sucursal_id,
                                'created_at'   => now(),
                                'updated_at'   => now(),
                            ]
                        );

                        PosBodegaArticuloPackItem::create([
                            'pba_pack_id'  => $pbaPackId,
                            'pba_item_id'  => $pbaItem->id,
                            'cantidad'     => $it['cantidad'],
                            'afecta_stock' => $it['afecta_stock'],
                            'estado'       => 1,
                            'created_at'   => now(),
                            'updated_at'   => now(),
                        ]);
                    }
                }
            }

            return $posArticulo->fresh();
        }

        // =========================
        // 5) VARIEDADES (exclusivo)
        // =========================
        if ($esVariedades) {

            // Normalizar variedades (hereda nombre del padre + presentación)
            $variedades = $variedadesInput->map(function ($v) use ($posArticulo, $request) {
                $presentacion = trim((string)($v['presentacion'] ?? $v['nombre'] ?? ''));
                $nombreFinal = $presentacion !== ''
                    ? ($posArticulo->nombre . ' - ' . $presentacion)
                    : ($posArticulo->nombre . ' / Variedad');

                return [
                    'nombre'        => $nombreFinal,
                    'venta'         => isset($v['venta']) ? (float)$v['venta'] : (float)($posArticulo->venta ?? 0),
                    'costo'         => isset($v['costo']) ? (float)$v['costo'] : (float)($request->costo ?? 0),
                    'barra'         => $v['barra'] ?? null,
                    'estado'        => isset($v['estado']) ? (int) $v['estado'] : (int)($request->estado ?? 1),
                    'stock_minimo'  => isset($v['stock_minimo']) ? (float) $v['stock_minimo'] : (float)($request->stock_minimo ?? 0),
                    'attrs_json'    => null,
                    'attrs_hash'    => null,
                ];
            })->values();

            foreach ($variedades as $v) {
                // Artículo variedad (child)
                $var = new \App\Models\Pos\PosArticulo();
                $var->tipo = 'variante';

                // Ajuste: setear el campo de relación correcto
                if (Schema::hasColumn('pos_articulos', 'parent_articulo_id')) {
                    $var->parent_articulo_id = $posArticulo->id;
                }
                if (Schema::hasColumn('pos_articulos', 'parent_id')) {
                    $var->parent_id = $posArticulo->id;
                }

                $var->nombre           = $v['nombre'];
                $var->sku              = null; // trigger genera
                $var->barra            = $v['barra'];
                $var->par_medida_id        = (int) $request->par_medida_id;
                $var->par_marca_id         = (int) $request->par_marca_id;
                $var->pos_categoria_id     = (int) $request->pos_categoria_id;
                $var->venta            = (float) $v['venta'];
                $var->peso             = (float) ($request->peso ?? 0);
                $var->stock_minimo     = (float) $v['stock_minimo'];
                $var->par_empresa_id       = (int) $request->par_empresa_id;
                $var->estado           = (int) $v['estado'];
                if (Schema::hasColumn('pos_articulos', 'atributos_json')) $var->atributos_json = $v['attrs_json'];
                if (Schema::hasColumn('pos_articulos', 'atributos_hash')) $var->atributos_hash = $v['attrs_hash'];
                $var->save();
                $var->refresh();

                // Impuestos heredados
                if (!empty($impuestosReq)) {
                    $rowsVarImp = array_map(fn ($i) => [
                        'pos_articulo_id'   => $var->id,
                        'con_impuesto_id'   => $i['con_impuesto_id'],
                        'activo'            => $i['activo'],
                        'optativo_en_venta' => $i['optativo_en_venta'],
                        'estado'            => 1,
                        'created_at'        => now(),
                        'updated_at'        => now(),
                    ], $impuestosReq);
                    PosArticuloImpuesto::insert($rowsVarImp);
                }

                // BA de variedad en todas las bodegas (enlazado al BA padre)
                foreach ($bodegas as $b) {
                    $baPadreId = $baPadrePorBodega[$b->id] ?? null;

                    // Fallback por si no estuviera set
                    if (!$baPadreId) {
                        $baPadreId = \App\Models\Pos\PosBodegaArticulo::where('pos_articulo_id', $posArticulo->id)
                            ->where('pos_bodega_id', $b->id)
                            ->value('id');
                    }

                    $baVar = PosBodegaArticulo::create([
                        'pos_articulo_id'           => $var->id,
                        'pos_bodega_id'             => $b->id,
                        'parent_bodega_articulo_id' => $baPadreId, // enlace al BA padre
                        'stock'                     => 0,
                        'c_stock'                   => (int) ($request->c_stock ?? 1),
                        'c_unitario'                => 0,
                        'p_venta'                   => (float) $v['venta'],
                        'costo'                     => (float) $v['costo'],
                        'par_empresa_id'                => (int) $request->par_empresa_id,
                        'par_sucursal_id'               => (int) $b->par_sucursal_id,
                        'estado'                    => (int) $v['estado'],
                        'created_at'                => now(),
                        'updated_at'                => now(),
                    ]);

                    // Impuestos a BA variedad
                    if (!empty($impuestosReq)) {
                        $rowsBAImp = array_map(fn ($i) => [
                            'pos_bodega_articulo_id' => $baVar->id,
                            'con_impuesto_id'        => $i['con_impuesto_id'],
                            'activo'                 => $i['activo'],
                            'optativo_en_venta'      => $i['optativo_en_venta'],
                            'created_at'             => now(),
                            'updated_at'             => now(),
                        ], $impuestosReq);
                        PosBodegaArticuloImpuesto::insert($rowsBAImp);
                    }
                }
            }

            return $posArticulo->fresh();
        }

        // =========================
        // 6) SIMPLE (sin pack ni variedades)
        // =========================
        return $posArticulo->fresh();
    });
}




    

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\PosArticulo  $posArticulo
     * @return \Illuminate\Http\Response
     */
    public function show(PosArticulo $posArticulo)
    {
        // Permitir ?with=impuestos,variantes,pack_items
        $with = request()->query('with', '');
        $requested = array_filter(array_map('trim', explode(',', $with)));
        $map = [
            'impuestos'  => 'impuestos.conImpuesto', // cargamos el impuesto padre
            'variantes'  => 'variantes',
            'pack_items' => 'pack_items',
            'bodegas'    => 'BodegaArticulos',
        ];

        $toLoad = [];
        foreach ($requested as $r) {
            if (isset($map[$r])) $toLoad[] = $map[$r];
        }

        if (!empty($toLoad)) {
            $posArticulo->loadMissing($toLoad);
        }

        // Si cargamos impuestos, inyectamos codigo/nombre del impuesto en cada fila
        if ($posArticulo->relationLoaded('impuestos')) {
            $posArticulo->setRelation('impuestos', $posArticulo->impuestos->map(function ($pai) {
                $arr = $pai->toArray();
                $arr['codigo'] = $pai->conImpuesto->codigo ?? null;
                $arr['nombre'] = $pai->conImpuesto->nombre ?? null;
                return $arr;
            }));
        }

        // Normalizamos la respuesta para que el frontend reciba { articulo: ... }
        return response()->json(['articulo' => $posArticulo]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\PosArticulo  $posArticulo
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, PosArticulo $posArticulo)
{
    return DB::transaction(function () use ($request, $posArticulo) {

        /* ========= 0) Normalización ========= */
        $empresaId = (int) ($request->par_empresa_id ?? $posArticulo->par_empresa_id);

        /* ========= 1) ACTUALIZA ARTÍCULO PRINCIPAL ========= */
        $posArticulo->tipo         = $posArticulo->tipo; // no lo cambiamos aquí
        $posArticulo->nombre       = $request->nombre;
        $posArticulo->sku          = $request->sku ?: null;
        $posArticulo->barra        = $request->barra ?: null;
        $posArticulo->par_medida_id    = (int) $request->par_medida_id;
        $posArticulo->par_marca_id     = (int) $request->par_marca_id;
        $posArticulo->pos_categoria_id = (int) $request->pos_categoria_id;
        $posArticulo->venta        = (float) ($request->venta ?? 0);
        $posArticulo->peso         = (float) ($request->peso ?? 0);
        $posArticulo->stock_minimo = (float) ($request->stock_minimo ?? 0);
        $posArticulo->par_empresa_id   = $empresaId;
        // permitir actualizar estado del artículo si viene en el request
        if ($request->has('estado')) {
            $posArticulo->estado = (int) $request->input('estado');
        }
        $posArticulo->save();

        /* ========= 2) Asegura BA del principal en TODAS las bodegas ========= */
        // Igual que en store(): todas las bodegas de la empresa
    $sucursalesIds = ParSucursal::where('par_empresa_id', $empresaId)->pluck('id');
    $bodegas       = PosBodega::whereIn('par_sucursal_id', $sucursalesIds)->get(['id','par_sucursal_id']);

        // Mapa: pos_bodega_id -> id del BA del principal
        $baPadrePorBodega = [];

        foreach ($bodegas as $b) {
            $ba = PosBodegaArticulo::firstOrCreate(
                [
                    'pos_articulo_id' => $posArticulo->id,
                    'pos_bodega_id'   => $b->id,
                ],
                [
                    'parent_bodega_articulo_id' => null,
                    'stock'       => 0,
                    'c_stock'     => (int) ($request->c_stock ?? 1),
                    'c_unitario'  => 0,
                    'p_venta'     => (float) ($request->venta ?? 0),
                    'costo'       => (float) ($request->costo ?? 0),
                    'par_empresa_id'  => $empresaId,
                    'par_sucursal_id' => (int) $b->par_sucursal_id,
                ]
            );

            // si quieres actualizar p_venta/costo/c_stock cuando ya existe, hazlo aquí
            $ba->p_venta = (float) ($request->venta ?? $ba->p_venta);
            $ba->c_stock = (int)   ($request->c_stock ?? $ba->c_stock);
            // permitir actualizar estado del BA si viene en el request
            if ($request->has('estado')) {
                $ba->estado = (int) $request->input('estado');
            }
            $ba->save();

            $baPadrePorBodega[$b->id] = $ba->id;
        }

        /* ========= 3) IMPUESTOS (UPSERT + Clonado a BA) ========= */
        $impuestosReq = collect($request->input('impuestos', []))
            ->filter(fn ($i) => is_array($i) && !empty($i['con_impuesto_id']))
            ->unique('con_impuesto_id');

        foreach ($impuestosReq as $row) {
            $conImpuestoId = (int) $row['con_impuesto_id'];
            $activo        = (int) ($row['activo'] ?? 1);
            $optativo      = (int) ($row['optativo_en_venta'] ?? 0);
            $relacionId    = $row['id'] ?? null;

            if ($relacionId) {
                // Actualiza si existe; sino, fallback a par (articulo,impuesto)
                $pai = PosArticuloImpuesto::where('id', $relacionId)
                    ->where('pos_articulo_id', $posArticulo->id)
                    ->first();

                if ($pai) {
                    // si NO quieres permitir cambiar el impuesto en edición, comenta la línea de abajo
                    $pai->con_impuesto_id   = $conImpuestoId;
                    $pai->activo            = $activo;
                    $pai->optativo_en_venta = $optativo;
                    $pai->save();
                } else {
                    $pai = PosArticuloImpuesto::updateOrCreate(
                        ['pos_articulo_id' => $posArticulo->id, 'con_impuesto_id' => $conImpuestoId],
                        ['activo' => $activo, 'optativo_en_venta' => $optativo]
                    );
                }
            } else {
                // Crear/actualizar por par
                $pai = PosArticuloImpuesto::updateOrCreate(
                    ['pos_articulo_id' => $posArticulo->id, 'con_impuesto_id' => $conImpuestoId],
                    ['activo' => $activo, 'optativo_en_venta' => $optativo]
                );
            }

            // Propagar a todas las bodegas del principal
            foreach ($baPadrePorBodega as $pbaId) {
                PosBodegaArticuloImpuesto::updateOrCreate(
                    ['pos_bodega_articulo_id' => $pbaId, 'con_impuesto_id' => $conImpuestoId],
                    ['activo' => $activo, 'optativo_en_venta' => $optativo]
                );
            }
        }

        /* ========= 4) VARIANTES (sub-artículos) ========= */
        $variedades = collect($request->input('variedades', []))
            ->filter(fn ($v) => is_array($v) && !empty($v['presentacion']));

        foreach ($variedades as $v) {
            $nombreVar = trim(($request->nombre ?? '').' - '.$v['presentacion']);

            $payloadVar = [
                'tipo'               => 'variante',
                'parent_articulo_id' => $posArticulo->id,
                'nombre'             => $nombreVar,
                'sku'                => $v['sku']   ?? null,
                'barra'              => $v['barra'] ?? null,
                'par_medida_id'          => (int) $request->par_medida_id,
                'par_marca_id'           => (int) $request->par_marca_id,
                'pos_categoria_id'       => (int) $request->pos_categoria_id,
                'venta'              => (float) ($v['venta'] ?? $request->venta ?? 0),
                'peso'               => (float) ($request->peso ?? 0),
                'stock_minimo'       => (float) ($v['stock_minimo'] ?? 0),
                'par_empresa_id'         => $empresaId,
                // si la variación trae estado explícito lo aplicamos
                'estado'             => isset($v['estado']) ? (int) $v['estado'] : ($request->has('estado') ? (int)$request->input('estado') : 1),
            ];

            if (!empty($v['id'])) {
                // UPDATE variación existente (si cuelga del padre)
                $artVar = PosArticulo::where('id', $v['id'])
                    ->where('parent_articulo_id', $posArticulo->id)
                    ->first();

                if ($artVar) {
                    $artVar->fill($payloadVar);
                    $artVar->save();
                } else {
                    // fallback: crea si no existe o no pertenece al padre
                    $artVar = PosArticulo::create($payloadVar);
                }
            } else {
                // CREATE nueva variación
                $artVar = PosArticulo::create($payloadVar);
            }

            // Impuestos de la variación = copiar del principal (igual que en store)
            $impuestosDelPadre = PosArticuloImpuesto::where('pos_articulo_id', $posArticulo->id)->get();
            foreach ($impuestosDelPadre as $i) {
                PosArticuloImpuesto::updateOrCreate(
                    ['pos_articulo_id' => $artVar->id, 'con_impuesto_id' => $i->con_impuesto_id],
                    ['activo' => $i->activo, 'optativo_en_venta' => $i->optativo_en_venta]
                );
            }

            // BA de la variación en TODAS las bodegas, apuntando al BA del padre
            foreach ($bodegas as $b) {
                $pbaPadreId = $baPadrePorBodega[$b->id] ?? null;

                $pbaVar = PosBodegaArticulo::updateOrCreate(
                    ['pos_articulo_id' => $artVar->id, 'pos_bodega_id' => $b->id],
                    [
                        'parent_bodega_articulo_id' => $pbaPadreId,
                        'stock'       => 0,
                        'c_stock'     => (int) ($request->c_stock ?? 1),
                        'c_unitario'  => 0,
                        'p_venta'     => (float) ($artVar->venta ?? 0),
                        'costo'       => (float) ($request->costo ?? 0),
                        'par_empresa_id'  => $empresaId,
                        'par_sucursal_id' => (int) $b->par_sucursal_id,
                            // aplicar estado en BA de la variante: si la variación trae estado lo usamos,
                            // sino usamos el estado del request si viene, o dejamos el existente
                            'estado'      => isset($v['estado']) ? (int)$v['estado'] : ($request->has('estado') ? (int)$request->input('estado') : null),
                    ]
                );

                // impuestos en el BA de la variación
                foreach ($impuestosDelPadre as $i) {
                    PosBodegaArticuloImpuesto::updateOrCreate(
                        ['pos_bodega_articulo_id' => $pbaVar->id, 'con_impuesto_id' => $i->con_impuesto_id],
                        ['activo' => $i->activo, 'optativo_en_venta' => $i->optativo_en_venta]
                    );
                }
            }
        }

        /* ========= 5) PACK ITEMS (maestro + por bodega) =========
           Tablas/columns exactas como en store:
           - pos_articulo_pack_items:  pack_id, item_articulo_id, cantidad, afecta_stock
           - pos_bodega_articulos_pack_items: pba_pack_id, pba_item_id, cantidad, afecta_stock
        */
        $packItems = collect($request->input('pack_items', []))
            ->filter(fn ($it) => is_array($it) && !empty($it['item_articulo_id']))
            ->values();

        if ($packItems->isNotEmpty()) {
            // Aseguramos BA pack del padre por bodega ya con $baPadrePorBodega

            foreach ($packItems as $it) {
                $payload = [
                    'pack_id'          => $posArticulo->id,
                    'item_articulo_id' => (int) $it['item_articulo_id'],
                    'cantidad'         => (float) ($it['cantidad'] ?? 1),
                    'afecta_stock'     => (int)   ($it['afecta_stock'] ?? 1),
                ];

                // Valores usados para upsert/creación (incluye estado si viene)
                $values = ['cantidad' => $payload['cantidad'], 'afecta_stock' => $payload['afecta_stock']];
                if (isset($it['estado'])) {
                    $values['estado'] = (int) $it['estado'];
                } elseif ($request->has('estado')) {
                    $values['estado'] = (int) $request->input('estado');
                }

                if (!empty($it['id'])) {
                    // UPDATE maestro
                    $pi = PosArticuloPackItem::where('id', $it['id'])
                        ->where('pack_id', $posArticulo->id)
                        ->first();

                    if ($pi) {
                        $pi->fill($payload);
                        // aplicar estado si viene
                        if (isset($it['estado'])) {
                            $pi->estado = (int) $it['estado'];
                        } elseif ($request->has('estado')) {
                            $pi->estado = (int) $request->input('estado');
                        }
                        $pi->save();
                    } else {
                        // fallback: upsert por (pack_id,item_articulo_id)
                        $pi = PosArticuloPackItem::updateOrCreate(
                            ['pack_id' => $posArticulo->id, 'item_articulo_id' => $payload['item_articulo_id']],
                            $values
                        );
                    }
                } else {
                    // CREATE maestro (o actualiza si ya existe par)
                    $pi = PosArticuloPackItem::updateOrCreate(
                        ['pack_id' => $posArticulo->id, 'item_articulo_id' => $payload['item_articulo_id']],
                        ['cantidad' => $payload['cantidad'], 'afecta_stock' => $payload['afecta_stock']]
                    );
                }

                // === Reflejo por bodega ===
                foreach ($bodegas as $b) {
                    $pbaPackId = $baPadrePorBodega[$b->id] ?? null;
                    if (!$pbaPackId) continue;

                    // Asegurar que exista BA del item en esa bodega
                    $pbaItem = PosBodegaArticulo::firstOrCreate(
                        [
                            'pos_articulo_id' => $payload['item_articulo_id'],
                            'pos_bodega_id'   => $b->id,
                        ],
                        [
                            'parent_bodega_articulo_id' => null,
                            'stock'       => 0,
                            'c_stock'     => 1,
                            'c_unitario'  => 0,
                            'p_venta'     => 0,
                            'costo'       => 0,
                            'par_empresa_id'  => $empresaId,
                            'par_sucursal_id' => (int) $b->par_sucursal_id,
                        ]
                    );

                    $valuesPba = ['cantidad' => $payload['cantidad'], 'afecta_stock' => $payload['afecta_stock']];
                    if (isset($it['estado'])) {
                        $valuesPba['estado'] = (int) $it['estado'];
                    } elseif ($request->has('estado')) {
                        $valuesPba['estado'] = (int) $request->input('estado');
                    }

                    PosBodegaArticuloPackItem::updateOrCreate(
                        ['pba_pack_id' => $pbaPackId, 'pba_item_id' => $pbaItem->id],
                        $valuesPba
                    );
                }
            }
        }

        /* ========= 6) RESPUESTA ========= */
        return response()->json([
            'ok'       => true,
            'articulo' => $posArticulo->fresh(),
        ]);
    });
}



    public function pventa(PosArticulo $posArticulo)
    {
        $posArticulo->save();
        return $posArticulo;
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Articulo  $articulo
     * @return \Illuminate\Http\Response
     */
    public function destroy(PosArticulo $posArticulo)
    {
        $posArticulo->estado = 0;
        $posArticulo->save();
        return $posArticulo;
    }
    public function Imagen(PosArticulo $posArticulo)
     {

         $posArticulo->marca = $posArticulo->Marca;
         $posArticulo->medida = $posArticulo->Medida;
         $posArticulo->bodega = $posArticulo->Bodega;
         $posArticulo->categoria = $posArticulo->Categoria;
         $posArticulo->image = $posArticulo->ArticuloImages()->get()->first();
         if($posArticulo->image!=null){
            $posArticulo->image->url = $posArticulo->image->image->UrlImage();
         }
         return $posArticulo;
     }
}
