<?php

namespace App\Http\Controllers\Pos;

use App\Models\Parametros\ParEmpresa as ParEmpresa;
use App\Models\Pos\PosCompra;
use Illuminate\Http\Request;
use App\Models\Pos\PosInventario;
use App\Models\Pos\PosCompraDetalle;
use App\Models\Pos\PosBodegaArticulo;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controller;

class PosCompraController extends Controller
{
    public function index()
    {
        $desde    = request()->get('desde');
        $hasta    = request()->get('hasta');
        $empresa  = request()->get('empresa');
        $sucursal = request()->get('sucursal');

        $query = PosCompra::where('id', '>', 0)->orderBy('fecha', 'desc');

        if ($empresa) {
            $query->where('par_empresa_id', $empresa);
        }

        if ($sucursal) {
            $query->where('par_sucursal_id', $sucursal);
        }

        if ($desde && $hasta) {
            $query->whereBetween('fecha', [$desde, $hasta]);
        } elseif ($desde) {
            $query->where('fecha', '>=', $desde);
        } elseif ($hasta) {
            $query->where('fecha', '<=', $hasta);
        } else {
            $query->whereDate('fecha', now());
        }

        $compras = $query->get();

        $list = [];
        foreach ($compras as $compra) {
            $list[] = $this->show($compra);
        }

        return $list;
    }

    public function store(Request $request)
{
    try {
        DB::beginTransaction();

        // ✅ 0) Método de inventario por ID (1=FIFO, 2=LIFO, 3=PROMEDIO, 4=IDENTIFICACION)
                $metodoId = (int) (ParEmpresa::whereKey($request->empresa_id)
                    ->value('metodo_inventario_id') ?: 3); // default PROMEDIO (3)

        // ✅ 1) Cabecera de compra
        $posCompra = PosCompra::create([
            'pos_orden_id'      => null,
            'fecha'             => $request->fecha,
            'con_doc_tributario_id' => $request->documento_id,
            'ndocumento'        => $request->ndocumento,
            'pos_metodo_id'         => $request->pos_metodo_id,
                'par_empresa_id'    => $request->par_empresa_id,
            'par_sucursal_id'       => $request->par_sucursal_id,
            'observacion'       => $request->observacion ?? '',
            'pos_proveedor_id'      => $request->pos_proveedor_id,
            'neto'              => $request->t_neto,
            'impuesto'          => $request->t_impuesto,
            'total'             => $request->total,
            'usuario_id'        => $request->usuario_id,
        ]);

        // ✅ 2) Items (fail-fast si no hay carrito)
        $items = $request->carrito ?? [];
        if (empty($items)) {
            throw new \Exception('El carrito de compra está vacío.');
        }

        foreach ($items as $item) {

            $bodegaArticulo = PosBodegaArticulo::where('id', $item['articulo']['id'])
                ->where('pos_bodega_id', $item['pos_bodega_id'])
                ->lockForUpdate()
                ->first();

            if (!$bodegaArticulo) {
                throw new \Exception("No se encontró el artículo en la bodega especificada");
            }

            $cantidad      = (float) ($item['cantidad'] ?? 0);
            if ($cantidad <= 0) {
                throw new \Exception("Cantidad inválida para el artículo {$item['articulo']['id']}");
            }
            $controlaStock = (int) ($item['c_stock'] ?? 1);

            // IVA: soporta 1.19 o 0.19
            $p         = (float) ($item['porcentaje'] ?? 0);
            $ivaFactor = $p > 1 ? $p : (1 + $p);   // 1.19
            $ivaTasa   = $p > 1 ? ($p - 1) : $p;   // 0.19

            // ⚠️ Si $item['precio'] ya es NETO, déjalo tal cual.
            // Si fuese BRUTO, conviértelo así: $precioCompra = ((float)$item['precio']) / $ivaFactor;
            $precioCompra = (float) $item['precio'];        // tratamos como NETO unitario
            $netoItem     = $precioCompra * $cantidad;
            $impuestoItem = $netoItem * $ivaTasa;
            $totalItem    = $netoItem + $impuestoItem;

            // ✅ 3) Movimiento en pos_inventarios (ENTRADA)
            $posInventario = PosInventario::create([
                'movimiento'             => 1, // forzado entrada
                'fecha'                  => $request->fecha,
                'pos_bodega_articulo_id' => $item['articulo']['id'],
                'texto'                  => $request->texto ?? '',
                'pos_bodega_id'          => $item['pos_bodega_id'],
                'c_stock'                => $controlaStock,
                'cantidad'               => $cantidad,
                'compra'                 => $precioCompra,                       // costo unitario neto
                'venta'                  => (float) ($item['pventa'] ?? 0),      // PVP sugerido (opcional)
                'neto'                   => $netoItem,
                'impuesto'               => $impuestoItem,
                'total'                  => $totalItem,
                'consumido'              => 0, // 👈 clave para FIFO/LIFO/Identificación

            ]);

            // ✅ 4) Detalle de compra
            PosCompraDetalle::create([
                'pos_inventario_id'      => $posInventario->id,
                'pos_compra_id'          => $posCompra->id,
                'pos_bodega_articulo_id' => $item['articulo']['id'],
                'texto'                  => $request->texto ?? '',
                'c_stock'                => $controlaStock,
                'pos_bodega_id'          => $item['pos_bodega_id'],
                'cantidad'               => $cantidad,
                'unitario'               => $precioCompra,
                'impuesto'               => $impuestoItem,
                'neto'                   => $netoItem,
                'total'                  => $totalItem,
            ]);

            // ✅ 5) Actualizar pos_bodega_articulos según el método (por ID)
            if ($controlaStock === 1) {
                if ($metodoId === 3) { // PROMEDIO (promedio móvil)
                    $costoActual        = (float) ($bodegaArticulo->costo ?? 0); // costo total actual
                    $stockActual        = (float) ($bodegaArticulo->stock ?? 0);
                    $nuevoCostoTotal    = $costoActual + $netoItem;
                    $nuevoStock         = $stockActual + $cantidad;
                    $nuevoCostoUnitario = $nuevoStock > 0 ? $nuevoCostoTotal / $nuevoStock : 0;

                    $bodegaArticulo->update([
                        'costo'      => $nuevoCostoTotal,
                        'stock'      => $nuevoStock,
                        'c_unitario' => $nuevoCostoUnitario,
                        'p_venta'    => (float) ($item['pventa'] ?? $bodegaArticulo->p_venta),
                    ]);

                } else { // 1=FIFO, 2=LIFO, 4=IDENTIFICACION
                    // En estos métodos, el costo real se usa al vender (consumiendo lotes).
                    // Aquí solo reflejamos el stock total y sumamos costo total (informativo).
                    $nuevoStock = (float) ($bodegaArticulo->stock ?? 0) + $cantidad;
                    $nuevoCosto = (float) ($bodegaArticulo->costo ?? 0) + $netoItem;

                    $bodegaArticulo->update([
                        'stock'   => $nuevoStock,
                        'costo'   => $nuevoCosto, // informativo
                        'p_venta' => (float) ($item['pventa'] ?? $bodegaArticulo->p_venta),
                        // c_unitario puede quedar como dato informativo (no se usa para costear salidas en FIFO/LIFO)
                    ]);
                }
            }
        }

        // (Opcional) Si manejas pagos de compra, itera sobre ($request->pagos ?? []) aquí…

        DB::commit();

        return response()->json([
            'success' => true,
            'compra'  => $posCompra,
            'message' => 'Compra guardada exitosamente',
        ]);

    } catch (\Exception $e) {
        DB::rollBack();
        return response()->json([
            'success' => false,
            'message' => 'Error al guardar la compra: ' . $e->getMessage(),
        ], 500);
    }
}


    public function show(PosCompra $posCompra)
    {
    // Cargar método de pago y proveedor para que estén disponibles en la respuesta
    $compra = PosCompra::with(['Metodo', 'Proveedor'])->where('id', $posCompra->id)->first();

        if (!$compra) return null;

        // Cargar detalles
        $detalles = PosCompraDetalle::where('pos_compra_id', $compra->id)->get();

        $compra->detalles = $detalles->map(function ($d) {
            return [
                'id'                    => $d->id,
                'pos_inventario_id'     => $d->pos_inventario_id,
                'pos_bodega_articulo_id'=> $d->pos_bodega_articulo_id,
                'pos_bodega_id'         => $d->pos_bodega_id,
                'texto'                 => $d->texto,
                'c_stock'               => $d->c_stock,
                'cantidad'              => $d->cantidad,
                'unitario'              => $d->unitario,
                'neto'                  => $d->neto,
                'impuesto'              => $d->impuesto,
                'total'                 => $d->total,
            ];
        });

        return $compra;
    }

    public function update(Request $request, PosCompra $posCompra)
    {
        //
    }

    public function destroy(PosCompra $posCompra)
    {
        //
    }
}
