<?php

namespace App\Http\Controllers\Pos;

use App\Models\Parametros\ParEmpresa as ParEmpresa;
use App\Models\Pos\PosTraspaso;
use Illuminate\Support\Arr;
use Illuminate\Http\Request;
use App\Models\Pos\PosInventario;
use Illuminate\Support\Carbon;
use App\Models\Pos\PosBodegaArticulo;
use App\Models\Pos\PosTraspasoDetalle;
use App\Models\Pos\PosTraspasoUsuario;
use Illuminate\Support\Facades\DB;
use App\Services\InventoryCostService;
use Illuminate\Routing\Controller;

class PosTraspasoController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        //
    }

    public function filtroDespacho(Request $request)
    {
        $request->validate([
            'par_empresa_id' => 'required|integer',
            'par_sucursal_id' => 'nullable|integer',
            'limit' => 'nullable|integer|min:1|max:100'
        ]);

        $parEmpresaId = (int) $request->par_empresa_id;
        $sucursalId = $request->filled('par_sucursal_id') ? (int) $request->par_sucursal_id : null;
        $limit = $request->get('limit', 50);

        // Columns requested
        $columns = [
            'id', 'folio', 'par_empresa_id', 'fecha_creacion', 'fecha_envio', 'fecha_eta', 'fecha_recepcion',
            'referencia', 'c_conductor', 'patente', 'fono', 'observacion', 'usuario_id', 'e_traspaso', 'estado', 'created_at', 'updated_at'
        ];
        $query = PosTraspaso::select($columns)
            ->where('par_empresa_id', $parEmpresaId)
            ->where('estado', 1)
            ->where('e_traspaso', 1);

        if ($sucursalId !== null) {
            // Filtrar traspasos cuya cabecera tiene par_sucursal_id = $sucursalId
            // OR cuyos detalles tienen una bodega destino cuya par_sucursal_id coincide.
            // Esto cubre ambos casos: la sucursal puede estar en la cabecera o en la bodega destino.
            $query->where(function ($qMain) use ($sucursalId) {
                $qMain->where('par_sucursal_id', $sucursalId)
                    ->orWhereExists(function ($q) use ($sucursalId) {
                        $q->select(DB::raw('1'))
                          ->from('pos_traspaso_detalles')
                          ->join('pos_bodegas', 'pos_traspaso_detalles.pos_bodega_destino_id', '=', 'pos_bodegas.id')
                          ->whereColumn('pos_traspaso_detalles.pos_traspaso_id', 'pos_traspasos.id')
                          ->where('pos_bodegas.par_sucursal_id', $sucursalId);
                    });
            });
        }

        $traspasos = $query->orderBy('fecha_creacion', 'desc')
            ->limit($limit)
            ->get();

        // eager load detalles and PBA relations with articulo name
        $traspasos->loadMissing([
            'pos_traspaso_detalles.pos_bodega_articulo.articulo',
            'pos_traspaso_detalles.pos_bodega_articulo.bodega',
            'pos_traspaso_detalles.pos_bodega_origen.articulo',
            'pos_traspaso_detalles.pos_bodega_origen.bodega',
            'pos_traspaso_detalles.pos_bodega_destino.articulo',
            'pos_traspaso_detalles.pos_bodega_destino.bodega',
        ]);

        // map detalles to include a convenient articulo_nombre (prefer main pba.articulo->nombre)
        $traspasos->transform(function ($t) {
            if ($t->relationLoaded('pos_traspaso_detalles')) {
                $t->pos_traspaso_detalles->transform(function ($d) {
                    $nombre = null;
                    if ($d->pos_bodega_articulo && $d->pos_bodega_articulo->relationLoaded('articulo') && $d->pos_bodega_articulo->articulo) {
                        $nombre = $d->pos_bodega_articulo->articulo->nombre ?? null;
                    } elseif ($d->pos_bodega_origen && $d->pos_bodega_origen->relationLoaded('articulo') && $d->pos_bodega_origen->articulo) {
                        $nombre = $d->pos_bodega_origen->articulo->nombre ?? null;
                    } elseif ($d->pos_bodega_destino && $d->pos_bodega_destino->relationLoaded('articulo') && $d->pos_bodega_destino->articulo) {
                        $nombre = $d->pos_bodega_destino->articulo->nombre ?? null;
                    }
                    $d->articulo_nombre = $nombre;
                    return $d;
                });
            }
            return $t;
        });

        return response()->json(['data' => $traspasos]);
    }

    public function filtroRecepcion(Request $request)
    {
        $request->validate([
            'par_empresa_id' => 'required|integer',
            'par_sucursal_id' => 'nullable|integer',
            'limit' => 'nullable|integer|min:1|max:100'
        ]);

        $parEmpresaId = (int) $request->par_empresa_id;
        $sucursalId = $request->filled('par_sucursal_id') ? (int) $request->par_sucursal_id : null;
        $limit = $request->get('limit', 50);

        // Columns requested
        $columns = [
            'id', 'folio', 'par_empresa_id', 'fecha_creacion', 'fecha_envio', 'fecha_eta', 'fecha_recepcion',
            'referencia', 'c_conductor', 'patente', 'fono', 'observacion', 'usuario_id', 'e_traspaso', 'estado', 'created_at', 'updated_at'
        ];

        $query = PosTraspaso::select($columns)
            ->where('par_empresa_id', $parEmpresaId)
            ->where('estado', 1)
            ->whereIn('e_traspaso', [2, 3]);

        if ($sucursalId !== null) {
            $query->where('par_sucursal_id', $sucursalId);
        }

        $traspasos = $query->orderBy('fecha_creacion', 'desc')
            ->limit($limit)
            ->get();

        // eager load detalles and PBA relations with articulo name
        $traspasos->loadMissing(['pos_traspaso_detalles.pos_bodega_articulo.articulo',
                                 'pos_traspaso_detalles.pos_bodega_origen.articulo',
                                 'pos_traspaso_detalles.pos_bodega_destino.articulo']);

        // map detalles to include a convenient articulo_nombre (prefer main pba.articulo->nombre)
        $traspasos->transform(function ($t) {
            if ($t->relationLoaded('pos_traspaso_detalles')) {
                $t->pos_traspaso_detalles->transform(function ($d) {
                    $nombre = null;
                    if ($d->pos_bodega_articulo && $d->pos_bodega_articulo->relationLoaded('articulo') && $d->pos_bodega_articulo->articulo) {
                        $nombre = $d->pos_bodega_articulo->articulo->nombre ?? null;
                    } elseif ($d->pos_bodega_origen && $d->pos_bodega_origen->relationLoaded('articulo') && $d->pos_bodega_origen->articulo) {
                        $nombre = $d->pos_bodega_origen->articulo->nombre ?? null;
                    } elseif ($d->pos_bodega_destino && $d->pos_bodega_destino->relationLoaded('articulo') && $d->pos_bodega_destino->articulo) {
                        $nombre = $d->pos_bodega_destino->articulo->nombre ?? null;
                    }
                    $d->articulo_nombre = $nombre;
                    return $d;
                });
            }
            return $t;
        });

        return response()->json(['data' => $traspasos]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
   public function store(Request $request, InventoryCostService $costService)
    {
        $request->validate([
            'par_empresa_id'   => 'required|integer',
            'par_sucursal_id'  => 'nullable|integer',
            'usuario_id'   => 'required|integer',
            'fecha'        => 'nullable|date',
            'referencia'   => 'nullable|string',
            'detalle'      => 'nullable|string',
            'items'        => 'required|array|min:1',
            'items.*.pos_bodega_articulo_id' => 'required|integer',
            'items.*.bodega_origen'          => 'required|integer',
            'items.*.bodega_destino'         => 'required|integer',
            'items.*.sucursal_destino'       => 'nullable|integer',
            'items.*.cantidad'               => 'required|numeric|min:0.0001',
            'items.*.lote_id'                => 'nullable|integer',
        ]);

        try {
            DB::beginTransaction();

            $metodoId = (int) (ParEmpresa::whereKey($request->par_empresa_id)
                ->value('metodo_inventario_id') ?: 3);

            // Un traspaso por cada bodega destino
            $byDestino = [];
            foreach ($request->items as $row) {
                $dest = (int) $row['bodega_destino'];
                $byDestino[$dest] ??= [];
                $byDestino[$dest][] = $row;
            }

            $traspasosCreados = [];
            $detallesCreados  = [];

            foreach ($byDestino as $bodegaDestino => $itemsDestino) {
                $nextFolio = $this->siguienteFolioTraspaso((int)$request->par_empresa_id);

                $traspaso = PosTraspaso::create([
                    'par_empresa_id'  => $request->par_empresa_id,
                    'folio'           => $nextFolio,
                    'fecha_creacion'  => $request->fecha ?? now(),
                    'fecha_eta'       => null,
                    'fecha_recepcion' => null,
                    'referencia'      => $request->referencia ?? null,
                    'observacion'     => $request->detalle ?? null,
                    'usuario_id'      => $request->usuario_id,
                    'par_sucursal_id'     => $request->par_sucursal_id,
                ]);
                $traspasosCreados[] = $traspaso;

                PosTraspasoUsuario::create([
                    'pos_traspaso_id' => $traspaso->id,
                    'par_empresa_id'  => $request->par_empresa_id,
                    'user_id'         => $request->usuario_id,
                    'movimiento'      => 3, // creado/despachado
                ]);

                foreach ($itemsDestino as $row) {
                    $cantidad     = (float) $row['cantidad'];
                    $bodegaOrigen = (int)   $row['bodega_origen'];
                    $loteId       = isset($row['lote_id']) ? (int)$row['lote_id'] : null;

                    // PBA ORIGEN (lock & validate)
                    $origenPbaId = (int)$row['pos_bodega_articulo_id'];
                    $pbaOrigen = PosBodegaArticulo::lockForUpdate()
                        ->where('id', $origenPbaId)
                        ->where('pos_bodega_id', $bodegaOrigen)
                        ->first();

                    if (!$pbaOrigen) {
                        throw new \Exception("El pos_bodega_articulo_id {$origenPbaId} no existe o no pertenece a la bodega_origen {$bodegaOrigen}.");
                    }

                    $controlaStock = (int)($pbaOrigen->c_stock ?? 1);

                    // PBA DESTINO (mismo pos_articulo_id)
                    $posArticuloId = (int)$pbaOrigen->pos_articulo_id;
                    $pbaDestino = PosBodegaArticulo::lockForUpdate()
                        ->where('pos_articulo_id', $posArticuloId)
                        ->where('pos_bodega_id', $bodegaDestino)
                        ->first();

                    if (!$pbaDestino) {
                        $pbaDestino = PosBodegaArticulo::create([
                            'pos_articulo_id' => $posArticuloId,
                            'pos_bodega_id'   => $bodegaDestino,
                            'c_stock'         => $pbaOrigen->c_stock ?? 1,
                            'stock'           => 0,
                            'p_venta'         => $pbaOrigen->p_venta ?? 0,
                            'costo'           => 0,
                            'c_unitario'      => 0,
                            'par_empresa_id'  => $request->par_empresa_id,
                            'par_sucursal_id'     => $row['sucursal_destino'] ?? ($pbaOrigen->par_sucursal_id ?? null),
                        ]);
                    }

                    // Stock ORIGEN
                    if ($controlaStock === 1) {
                        $stockOrigen = (float)($pbaOrigen->stock ?? 0);
                        if ($stockOrigen < $cantidad) {
                            throw new \Exception("Stock insuficiente en bodega origen para el PBA {$origenPbaId}. Disp: {$stockOrigen}, req: {$cantidad}");
                        }
                    }

                    // Reserva en ORIGEN
                    if (array_key_exists('en_origen_in', $pbaOrigen->getAttributes())) {
                        $pbaOrigen->increment('en_origen_in', $cantidad);
                        $pbaOrigen->refresh();
                    }

                    // Costeo al salir
                    $costoUnit    = 0.0;
                    $loteRefId    = null;
                    $detalleLotes = null;

                    if ($controlaStock !== 1) {
                        // Para artículos que no controlan stock, tomar el unitario del PBA (si existe)
                        $costoUnit = (float)($pbaOrigen->c_unitario ?? 0);
                    } else {
                        switch ($metodoId) {
                            case 1:
                                $costoUnit = $costService->costearFIFO($origenPbaId, $bodegaOrigen, $cantidad);
                                break;
                            case 2:
                                $costoUnit = $costService->costearLIFO($origenPbaId, $bodegaOrigen, $cantidad);
                                break;
                            case 4:
                                if (empty($loteId)) {
                                    throw new \Exception("Para IDENTIFICACION debes indicar lote_id en la línea.");
                                }
                                $costoUnit = $costService->costearIdentificacion($loteId, $cantidad, $origenPbaId, $bodegaOrigen);
                                $loteRefId = $loteId;
                                $detalleLotes = json_encode([['lote_id' => $loteId, 'cantidad' => $cantidad]]);
                                break;
                            case 3:
                                $costoUnit = (float)($pbaOrigen->c_unitario ?? 0);
                                break;
                        }
                    }

                    // Detalle del traspaso
                    $det = PosTraspasoDetalle::create([
                        'pos_traspaso_id'        => $traspaso->id,
                        'pos_bodega_articulo_id' => $origenPbaId,
                        'pos_bodega_origen_id'   => $bodegaOrigen,
                        'pos_bodega_destino_id'  => $bodegaDestino,
                        'cantidad'               => $cantidad,
                        'cantidad_recibida'      => 0,
                        'costo_unitario_fijado'  => $costoUnit,
                        'par_empresa_id'  => $request->par_empresa_id,
                        'detalle_lotes'          => $detalleLotes,
                    ]);
                    $detallesCreados[] = $det;

                    // ORIGEN: mov=3 (salida)
                    $neto = $costoUnit * $cantidad;
                    PosInventario::create([
                        'movimiento'              => 3,
                        'fecha'                   => $traspaso->fecha_creacion ?? now(),
                        'pos_bodega_articulo_id'  => $origenPbaId,
                        'pos_bodega_id'           => $bodegaOrigen,
                        'c_stock'                 => $controlaStock,
                        'cantidad'                => $cantidad,
                        'compra'                  => $costoUnit,
                        'venta'                   => 0,
                        'neto'                    => $neto,
                        'impuesto'                => 0,
                        'total'                   => $neto,
                        'tipo'                    => 0,
                        'texto'                   => 'Traspaso salida',
                        'pos_traspaso_id'         => $traspaso->id,
                        'pos_traspaso_detalle_id' => $det->id,
                        'costo_fijado'            => $costoUnit,
                        'lote_ref_id'             => $loteRefId,
                    ]);

                    // PBA ORIGEN: rebaja stock y costo (con costo real de salida)
                    if ($controlaStock === 1) {
                        $stockActual       = (float)($pbaOrigen->stock ?? 0);
                        $nuevoStock        = $stockActual - $cantidad;
                        $costoTotalActual  = (float)($pbaOrigen->costo ?? 0);
                        if ($costoTotalActual <= 0 && $stockActual > 0) {
                            $costoTotalActual = ((float)($pbaOrigen->c_unitario ?? 0)) * $stockActual;
                        }
                        $rebajaCosto   = $costoUnit * $cantidad;
                        $nuevoCostoTot = max(0, $costoTotalActual - $rebajaCosto);
                        $nuevoCU       = $nuevoStock > 0 ? ($nuevoCostoTot / $nuevoStock) : 0;

                        $pbaOrigen->update([
                            'stock'      => $nuevoStock,
                            'costo'      => $nuevoCostoTot,
                            'c_unitario' => $nuevoCU,
                        ]);
                    }

                    // Limpia reserva en ORIGEN
                    if (array_key_exists('en_origen_in', $pbaOrigen->getAttributes())) {
                        $pbaOrigen->refresh();
                        $rebaja = min($cantidad, (float)($pbaOrigen->en_origen_in ?? 0));
                        if ($rebaja > 0) {
                            $pbaOrigen->decrement('en_origen_in', $rebaja);
                        }
                    }

                    // DESTINO: mov=4 (en tránsito)
                    PosInventario::create([
                        'movimiento'              => 4,
                        'fecha'                   => $traspaso->fecha_creacion ?? now(),
                        'pos_bodega_articulo_id'  => $pbaDestino->id,
                        'pos_bodega_id'           => $bodegaDestino,
                        'c_stock'                 => $controlaStock,
                        'cantidad'                => $cantidad,
                        'compra'                  => $costoUnit,
                        'venta'                   => 0,
                        'neto'                    => $neto,
                        'impuesto'                => 0,
                        'total'                   => $neto,
                        'tipo'                    => 0,
                        'texto'                   => 'Traspaso en origen',
                        'pos_traspaso_id'         => $traspaso->id,
                        'pos_traspaso_detalle_id' => $det->id,
                        'costo_fijado'            => $costoUnit,
                        'lote_ref_id'             => $loteRefId,
                    ]);

                    // marca tránsito en PBA destino (en_origen_in)
                    if (array_key_exists('en_origen_in', $pbaDestino->getAttributes())) {
                        $pbaDestino->increment('en_origen_in', $cantidad);
                    }
                }
            }

            DB::commit();

            return response()->json([
                'success'  => true,
                'message'  => 'Traspasos creados correctamente',
                'traspasos'=> collect($traspasosCreados)->map->only(['id','folio','estado','fecha_creacion']),
                'detalles' => $detallesCreados,
            ]);

        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al crear traspaso: '.$e->getMessage(),
            ], 500);
        }
    }

    /**
     * Genera el siguiente folio de TRASPASO por empresa con bloqueo.
     */
    private function siguienteFolioTraspaso(int $empresaId): int
    {
        try {
            $row = DB::table('pos_folios')
                ->where('par_empresa_id', $empresaId)
                ->where('nombre', 'TRASPASO')
                ->lockForUpdate()
                ->first();

            if (!$row) {
                $id = DB::table('pos_folios')->insertGetId([
                    'par_empresa_id' => $empresaId,
                    'nombre'      => 'TRASPASO',
                    'folio'       => 0,
                    'created_at'  => now(),
                    'updated_at'  => now(),
                ]);
                $row = DB::table('pos_folios')->where('id', $id)->lockForUpdate()->first();
            }

            $next = ((int)$row->folio) + 1;

            DB::table('pos_folios')
                ->where('id', $row->id)
                ->update(['folio' => $next, 'updated_at' => now()]);

            return $next;
        } catch (\Throwable $e) {
            $max = (int) DB::table('pos_traspasos')
                ->where('par_empresa_id', $empresaId)
                ->max('folio');
            return $max + 1;
        }
    }

    public function show(PosTraspaso $posTraspaso)
    {
        //
    }

    public function update(Request $request, PosTraspaso $posTraspaso)
{
    // Estados permitidos en este flujo
    $request->validate([
        'fecha_envio'   => 'nullable|date',    // usado en estado 2 (en tránsito)
        'fecha_recepcion'    => 'nullable|date',    // opcional para estado 3 (en destino)
        'fecha'            => 'nullable|date',    // compat: si te llega como "fecha" lo usamos para destino
        'fecha_recepcion'  => 'nullable|date',    // usado en estado 4 (recepcionado)
        'c_conductor'      => 'nullable|string|max:150',
        'patente'          => 'nullable|string|max:50',
        'fono'             => 'nullable|string|max:50',
        'e_traspaso'       => 'required_without:e_traspasos|integer|in:2,3,4',
        'e_traspasos'      => 'required_without:e_traspaso|integer|in:2,3,4',
    ]);

    $estado = (int) $request->input('e_traspaso', $request->input('e_traspasos'));

    // Normalización de fechas
    $fechaEnvio  = $request->filled('fecha_envio')
        ? \Carbon\Carbon::parse($request->input('fecha_envio'))->format('Y-m-d H:i:s')
        : now();

    // para "en destino" aceptamos fecha_recepcion o fecha (compat)
    $fechaDestino   = $request->filled('fecha_recepcion')
        ? \Carbon\Carbon::parse($request->input('fecha_recepcion'))->format('Y-m-d H:i:s')
        : ($request->filled('fecha')
            ? \Carbon\Carbon::parse($request->input('fecha'))->format('Y-m-d H:i:s')
            : now());

    $fechaRecepcion = $request->filled('fecha_recepcion')
        ? \Carbon\Carbon::parse($request->input('fecha_recepcion'))->format('Y-m-d H:i:s')
        : now();

    // ==========================
    // (2) EN TRÁNSITO (solo metadata, no mueve inventario)
    // ==========================
   if ($estado === 2) {

    DB::beginTransaction();
    try {
        // Traer detalles del traspaso
        $detalles = \App\Models\Pos\PosTraspasoDetalle::where('pos_traspaso_id', $posTraspaso->id)
            ->lockForUpdate()
            ->get();

        foreach ($detalles as $det) {

            $cantidad = (float) ($det->cantidad ?? 0);

            // ===== ORIGEN =====
            $pbaOrigen = \App\Models\Pos\PosBodegaArticulo::lockForUpdate()
                ->where('id', $det->pos_bodega_articulo_id)
                ->where('pos_bodega_id', $det->pos_bodega_origen_id)
                ->first();

            if ($pbaOrigen && array_key_exists('en_origen_in', $pbaOrigen->getAttributes())) {
                // dejar origen limpio
                $pbaOrigen->update(['en_origen_in' => 0]);
            }

            // ===== DESTINO =====
            $pbaDestino = \App\Models\Pos\PosBodegaArticulo::lockForUpdate()
                ->where('pos_articulo_id', $pbaOrigen->pos_articulo_id)
                ->where('pos_bodega_id', $det->pos_bodega_destino_id)
                ->first();

          if (
    $pbaDestino &&
    array_key_exists('en_transito_in', $pbaDestino->getAttributes()) &&
    array_key_exists('en_origen_in', $pbaDestino->getAttributes())
) {
    // 1️⃣ Suma a tránsito
    $pbaDestino->increment('en_transito_in', $cantidad);

    // 2️⃣ Deja origen en cero
    $pbaDestino->update([
        'en_origen_in' => 0
    ]);
}

            // ===== ACTUALIZAR TEXTO DEL MOVIMIENTO 4 =====
            \App\Models\Pos\PosInventario::where('pos_traspaso_id', $posTraspaso->id)
                ->where('pos_traspaso_detalle_id', $det->id)
                ->where('movimiento', 4)
                ->update([
                    'texto' => 'Traspaso en Transito'
                ]);
        }

        // Actualiza cabecera del traspaso
        DB::table('pos_traspasos')->where('id', $posTraspaso->id)->update([
            'fecha_envio' => $fechaEnvio,
            'e_traspaso'  => 2,
            'c_conductor' => $request->input('c_conductor'),
            'patente'     => $request->input('patente'),
            'fono'        => $request->input('fono'),
            'updated_at'  => now(),
        ]);

        DB::commit();
        return response()->json([
            'success' => true,
            'message' => 'Traspaso marcado En tránsito'
        ]);

    } catch (\Throwable $e) {
        DB::rollBack();
        return response()->json([
            'success' => false,
            'message' => $e->getMessage()
        ], 500);
    }
}


    // ==========================
    // (3) EN DESTINO (marca llegada/ETA, no mueve inventario)
    //   Nota: históricamente usábamos fecha_eta; aquí dejamos ambos por compat:
    //         fecha_recepcion y fecha_eta apuntan a la misma idea: llegó al destino pero sin recepcionar.
    // ==========================
if ($estado === 3) {

    DB::beginTransaction();
    try {
        // Actualiza cabecera
        DB::table('pos_traspasos')->where('id', $posTraspaso->id)->update([
            'fecha_eta'        => $fechaDestino,   // compat con versiones previas
            'fecha_recepcion'  => $fechaDestino,   // si existe esta columna
            'e_traspaso'       => 3,
            'c_conductor'      => $request->input('c_conductor'),
            'patente'          => $request->input('patente'),
            'fono'             => $request->input('fono'),
            'updated_at'       => now(),
        ]);

        // 🔹 Actualiza TEXTO de los movimientos del traspaso
        \App\Models\Pos\PosInventario::where('pos_traspaso_id', $posTraspaso->id)
            ->whereIn('movimiento', [3, 4])
            ->update([
                'texto' => 'Traspaso en Destino'
            ]);

        DB::commit();
        return response()->json([
            'success' => true,
            'message' => 'Traspaso marcado En destino'
        ]);

    } catch (\Throwable $e) {
        DB::rollBack();
        return response()->json([
            'success' => false,
            'message' => $e->getMessage()
        ], 500);
    }
}


    // ==========================
    // (4) RECEPCIONADO (aquí SÍ mueve inventario: ENTRADA mov=6)
    // ==========================
    if ($estado === 4) {

    DB::beginTransaction();
    try {
        // 🔒 Traemos y bloqueamos los detalles del traspaso
        $detalles = \App\Models\Pos\PosTraspasoDetalle::where('pos_traspaso_id', $posTraspaso->id)
            ->lockForUpdate()
            ->get();

        $movidos = [];

        foreach ($detalles as $det) {

            $cantidadReq = (float) ($det->cantidad ?? 0);
            $yaRecibida  = (float) ($det->cantidad_recibida ?? 0);
            $pendRecibir = max(0, $cantidadReq - $yaRecibida);

            if ($pendRecibir <= 0) {
                continue; // Nada pendiente
            }

            // ==========================
            // 🔍 Buscar movimiento EN TRÁNSITO (mov=4)
            // ==========================
            $movTransito = \App\Models\Pos\PosInventario::where('pos_traspaso_id', $posTraspaso->id)
                ->where('pos_traspaso_detalle_id', $det->id)
                ->where('movimiento', 4)
                ->lockForUpdate()
                ->first();

            if (!$movTransito) {
                throw new \Exception("No se encontró movimiento en tránsito para el detalle #{$det->id}");
            }

            // ==========================
            // 📦 PBA DESTINO
            // ==========================
            $pbaDestino = \App\Models\Pos\PosBodegaArticulo::lockForUpdate()->find(
                $movTransito->pos_bodega_articulo_id
            );

            if (!$pbaDestino) {
                throw new \Exception("No se encontró PBA destino para el detalle #{$det->id}");
            }

            $costoUnit = (float) ($det->costo_unitario_fijado ?? 0);
            $neto      = $costoUnit * $pendRecibir;

            // ==========================
            // 🔁 Convertir movimiento 4 → 6
            // ==========================
            $movTransito->update([
                'movimiento' => 6,
                'fecha'      => $fechaRecepcion,
                'cantidad'   => $pendRecibir,
                'neto'       => $neto,
                'total'      => $neto,
                'texto'      => 'Traspaso recepcionado',
            ]);

            // ==========================
            // 📈 Actualizar STOCK DESTINO
            // ==========================
            if ((int) ($pbaDestino->c_stock ?? 1) === 1) {

                $stockActual = (float) ($pbaDestino->stock ?? 0);
                $costoActual = (float) ($pbaDestino->costo ?? 0);

                $nuevoStock = $stockActual + $pendRecibir;
                $nuevoCosto = $costoActual + $neto;

                $nuevoCU = $nuevoStock > 0
                    ? $nuevoCosto / $nuevoStock
                    : 0;

                $pbaDestino->update([
                    'stock'      => $nuevoStock,
                    'costo'      => $nuevoCosto,
                    'c_unitario' => $nuevoCU,
                ]);
            }

            // ==========================
            // 🧹 Limpiar tránsito
            // ==========================
            if (array_key_exists('en_transito_in', $pbaDestino->getAttributes())) {
                $pbaDestino->decrement('en_transito_in', $pendRecibir);
            }

            // ==========================
            // ✅ Marcar detalle como recibido
            // ==========================
            $det->increment('cantidad_recibida', $pendRecibir);
            $movidos[] = $det->id;
        }

        // ==========================
        // 🏁 Marcar cabecera como recepcionada
        // ==========================
        DB::table('pos_traspasos')->where('id', $posTraspaso->id)->update([
            'fecha_recepcion' => $fechaRecepcion,
            'e_traspaso'      => 4,
            'updated_at'      => now(),
        ]);

        DB::commit();

        return response()->json([
            'success'  => true,
            'detalles' => $movidos,
            'message'  => 'Traspaso recepcionado correctamente'
        ]);

    } catch (\Throwable $e) {
        DB::rollBack();
        return response()->json([
            'success' => false,
            'message' => $e->getMessage()
        ], 500);
    }
}


    // Fallback (no debería ocurrir por la validación, pero dejamos actualización de metadatos)
    DB::table('pos_traspasos')->where('id', $posTraspaso->id)->update([
        'c_conductor' => $request->input('c_conductor'),
        'patente'     => $request->input('patente'),
        'fono'        => $request->input('fono'),
        'e_traspaso'  => $estado,
        'updated_at'  => now(),
    ]);

    return response()->json(['success' => true]);
}


    public function destroy(PosTraspaso $posTraspaso)
    {
        //
    }
}
