<?php

namespace App\Http\Controllers\Pos;

use App\Models\Parametros\ParEmpresa as ParEmpresa;
use App\Models\Pos\PosCaja;
use App\Models\Pos\PosVenta;
use Illuminate\Http\Request;
use App\Models\Pos\PosInventario;
use App\Models\Pos\PosVentaDetalle;
use App\Models\Pos\PosBodegaArticulo;
use App\Models\Pos\PosCajaMovimiento;
use Illuminate\Support\Facades\DB;
use App\Services\InventoryCostService;
use Illuminate\Routing\Controller;

class PosVentaController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */

    public function index(Request $request)
    {
        $desde = $request->desde;
        $hasta = $request->hasta;
        $empresa = $request->empresa;
        $sucursal = $request->sucursal;
        
        // ✅ Iniciar query base
        $query = PosVenta::where('estado', 1)
                        ->with(['Documento', 'Metodo', 'Cliente'])
                        ->orderBy('fecha', 'desc');
        
        // ✅ Filtrar por empresa si se proporciona
        if ($empresa) {
            $query->where('par_empresa_id', $empresa);
        }

        // ✅ Filtrar por sucursal si se proporciona
        if ($sucursal) {
            $query->where('par_sucursal_id', $sucursal);
        }
        
        // ✅ Filtrar por fechas
        if ($desde && $hasta) {
            // Si hay rango de fechas, usar el rango
            $query->whereBetween('fecha', [$desde, $hasta]);
        } elseif ($desde) {
            // Si solo hay fecha inicio, filtrar desde esa fecha
            $query->where('fecha', '>=', $desde);
        } elseif ($hasta) {
            // Si solo hay fecha fin, filtrar hasta esa fecha
            $query->where('fecha', '<=', $hasta);
        } else {
            // Si no hay fechas, mostrar solo el día actual
            $query->whereDate('fecha', now());
        }
        
        $ventas = $query->get();
        
        // ✅ Procesar cada venta con el método show
        $list = [];
        foreach ($ventas as $venta) {
            $list[] = $this->show($venta);
        }
        
        return $list;
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */

    
public function store(Request $request, InventoryCostService $costService)
{
    try {
        DB::beginTransaction();

        // ===== 0) Método de inventario
        $metodoId = (int) (ParEmpresa::whereKey($request->empresa_id)
            ->value('metodo_inventario_id') ?: 3);

        // ===== Config local
        $TIPO_VENTA    = (int)($request->tipo ?? 2); // salida por venta
        $TIPO_RESERVA  = 98;                         // marca "reserva de venta" (ajústalo si usas otro)

        // ===== 1) Totales de respaldo (por si el front no envía desglose completo)
        $sumNeto = $sumIva = $sumTotal = 0.0;
        foreach ((array) $request->carrito as $it) {
            $n = (float)($it['neto'] ?? 0);
            $e = (float)($it['exento'] ?? 0);
            $v = (float)($it['impuesto'] ?? 0);
            $t = (float)($it['total'] ?? 0);

            if (($n + $e + $v) <= 0 && $t > 0) {
                $tasa = (float)($it['tasa_iva_snap'] ?? 0);
                $afex = (string)($it['afecto_exento'] ?? 'afecto');
                if ($afex === 'afecto' && $tasa > 0) {
                    $n = round($t / (1 + $tasa));
                    $v = $t - $n;
                    $e = 0;
                } else {
                    $e = round($t); $n = 0; $v = 0;
                }
            }
            $sumNeto  += $n;
            $sumIva   += $v;
            $sumTotal += $t ?: ($n + $e + $v);
        }

        // ===== 2) Cabecera
        $posVenta = PosVenta::create([
            'orden'             => $request->orden,
            'fecha'             => $request->fecha,
            'par_empresa_id'    => $request->empresa_id,
            'par_sucursal_id'       => $request->sucursal_id,
            'observacion'       => $request->observacion ?? '',
            'track_id'          => null,
            'pdf_url'           => null,
            'pos_cliente_id'    => $request->pos_cliente_id,
            'con_doc_tributario_id' => $request->documento_id,
            'metodo_id'         => $request->metodo_id,
            'neto'              => round($request->t_neto     ?? $sumNeto),
            'impuesto'          => round($request->t_impuesto ?? $sumIva),
            'total'             => round($request->total      ?? $sumTotal),
            'usuario_id'        => $request->usuario_id,
        ]);

        // ===== 3) Ítems
        foreach ((array) $request->carrito as $item) {
            $cantidad      = (float) ($item['cantidad'] ?? 0);
            $articuloId    = (int)   ($item['articulo']['id'] ?? 0);     // 0 si es detalle libre
            $bodegaId      = (int)   ($item['articulo']['pos_bodega_id'] ?? 0);
            $controlaStock = (int)   ($item['c_stock'] ?? 0);

            $tasaPct      = (float) ($item['tasa_iva_snap'] ?? 0);
            $afectoExento = (string)($item['afecto_exento'] ?? ($tasaPct > 0 ? 'afecto' : 'exento'));

            $netoItem     = (float) ($item['neto']     ?? 0);
            $exentoItem   = (float) ($item['exento']   ?? 0);
            $impuestoItem = (float) ($item['impuesto'] ?? 0);
            $totalItem    = (float) ($item['total']    ?? 0);

            if (($netoItem + $exentoItem + $impuestoItem) <= 0 && $totalItem > 0) {
                if ($afectoExento === 'afecto' && $tasaPct > 0) {
                    $netoItem     = round($totalItem / (1 + $tasaPct));
                    $impuestoItem = $totalItem - $netoItem;
                    $exentoItem   = 0;
                } else {
                    $exentoItem   = round($totalItem);
                    $netoItem     = 0;
                    $impuestoItem = 0;
                }
            }

            // Unitario NETO para asentar (desde neto o exento)
            $unitarioNeto = 0.0;
            if ($cantidad > 0) {
                $base = $netoItem > 0 ? $netoItem : $exentoItem;
                $unitarioNeto = $base > 0 ? ($base / $cantidad) : 0;
            }

            $posInvIdFisico = null;

            // ===== 3.1) Ítem NO controlado o detalle libre => registra movimiento lógico (no descuenta stock)
            if ($articuloId === 0 || $controlaStock !== 1) {
                $textoBase = trim(($request->observacion ?? '').' VENTA (servicio/detalle sin stock)');
                $invLog = PosInventario::create([
                    'movimiento'             => 2,
                    'fecha'                  => $request->fecha,
                    'pos_bodega_articulo_id' => $articuloId ?: null,
                    'texto'                  => $textoBase,
                    'pos_bodega_id'          => $bodegaId ?: null,
                    'c_stock'                => 0,
                    'cantidad'               => $cantidad,
                    'compra'                 => 0,               // sin costo inventariable
                    'venta'                  => $unitarioNeto,
                    'neto'                   => round($netoItem),
                    'impuesto'               => round($impuestoItem),
                    'total'                  => round($totalItem),
                    'tipo'                   => $TIPO_VENTA,
                ]);
                $posInvIdFisico = $invLog->id; // linkeamos para trazabilidad
            }
            // ===== 3.2) Ítem con control de stock
            else {
                // Bodega/stock actual
                $ba = PosBodegaArticulo::lockForUpdate()->findOrFail($articuloId);
                $stock      = (float)($ba->stock ?? 0);
                $enOrigen   = (float)($ba->en_origen_in ?? 0);
                $enTransito = (float)($ba->en_transito_in ?? 0);
                $comprom    = (float)($ba->comprometido ?? 0);

                $dispFis  = $stock;
                $dispVirt = max(0, $stock + $enOrigen + $enTransito - $comprom);

                if ($cantidad > $dispVirt) {
                    throw new \Exception("Stock insuficiente para el artículo {$articuloId} en bodega {$bodegaId}. Físico: {$dispFis}, Virtual: {$dispVirt}");
                }

                $qtyFromStock = min($cantidad, $dispFis);          // sale ahora
                $qtyToReserve = max(0, $cantidad - $qtyFromStock); // queda reservado

                // --- a) SALIDA física (si hay)
                if ($qtyFromStock > 0) {
                    // Costeo salida física
                    switch ($metodoId) {
                        case 1: $cu = $costService->costearFIFO($articuloId, $bodegaId, $qtyFromStock); break;
                        case 2: $cu = $costService->costearLIFO($articuloId, $bodegaId, $qtyFromStock); break;
                        case 4:
                            if (empty($item['lote_id'])) {
                                throw new \Exception('Falta lote_id para Identificación específica');
                            }
                            $loteOk = PosInventario::where('id', (int)$item['lote_id'])
                                ->where('movimiento', 1)
                                ->where('pos_bodega_articulo_id', $articuloId)
                                ->where('pos_bodega_id', $bodegaId)
                                ->lockForUpdate()
                                ->exists();
                            if (!$loteOk) { throw new \Exception('Lote inválido para este artículo/bodega'); }
                            $cu = $costService->costearIdentificacion((int)$item['lote_id'], (float)$qtyFromStock);
                            break;
                        case 3:
                        default:
                            $cu = (float) ($ba->c_unitario ?? 0);
                            break;
                    }

                    $ratio   = $cantidad > 0 ? ($qtyFromStock / $cantidad) : 0;
                    $netoNow = round($netoItem     * $ratio);
                    $ivaNow  = round($impuestoItem * $ratio);
                    $totNow  = round($totalItem    * $ratio);

                    $invFis = PosInventario::create([
                        'movimiento'             => 2, // salida
                        'fecha'                  => $request->fecha,
                        'pos_bodega_articulo_id' => $articuloId,
                        'texto'                  => (($request->observacion ?? '') ? $request->observacion.' | ' : '') . 'VENTA (salida física)',
                        'pos_bodega_id'          => $bodegaId,
                        'c_stock'                => 1,
                        'cantidad'               => $qtyFromStock,
                        'compra'                 => $cu,
                        'venta'                  => $unitarioNeto,
                        'neto'                   => $netoNow,
                        'impuesto'               => $ivaNow,
                        'total'                  => $totNow,
                        'tipo'                   => $TIPO_VENTA,
                    ]);
                    $posInvIdFisico = $invFis->id;

                    // Descuento de stock/costos
                    if ($metodoId === 3) {
                        $costoActual   = (float) ($ba->costo ?? 0);
                        $nuevoStock    = $stock - $qtyFromStock;
                        $nuevoCostoTot = $costoActual - ($ba->c_unitario * $qtyFromStock);
                        $nuevoCu       = $nuevoStock > 0 ? ($nuevoCostoTot / $nuevoStock) : 0;

                        $ba->update([
                            'stock'      => $nuevoStock,
                            'costo'      => $nuevoCostoTot,
                            'c_unitario' => $nuevoCu,
                        ]);
                    } else {
                        $ba->update(['stock' => $stock - $qtyFromStock]);
                    }
                }

                // --- b) RESERVA (si falta parte) — NO descuenta stock, guarda costo provisorio
                if ($qtyToReserve > 0) {
                    $ratio   = $cantidad > 0 ? ($qtyToReserve / $cantidad) : 0;
                    $netoRes = round($netoItem     * $ratio);
                    $ivaRes  = round($impuestoItem * $ratio);
                    $totRes  = round($totalItem    * $ratio);

                    // Costo estimado (para márgenes/reportes) hasta que llegue físicamente
                    $cuEstimado = $costService->estimarCostoUnitario($articuloId, $bodegaId);

                    PosInventario::create([
                        'movimiento'             => 2,
                        'fecha'                  => $request->fecha,
                        'pos_bodega_articulo_id' => $articuloId,
                        'texto'                  => (($request->observacion ?? '') ? $request->observacion.' | ' : '') . 'RESERVA VENTA (no descuenta stock, costo provisorio)',
                        'pos_bodega_id'          => $bodegaId,
                        'c_stock'                => 0,               // clave: no descuenta stock
                        'cantidad'               => $qtyToReserve,
                        'compra'                 => $cuEstimado,     // << ahora guardamos costo estimado
                        'venta'                  => $unitarioNeto,
                        'neto'                   => $netoRes,
                        'impuesto'               => $ivaRes,
                        'total'                  => $totRes,
                        'tipo'                   => $TIPO_RESERVA,
                    ]);

                    // Marcar reservado
                    $ba->update([
                        'comprometido' => $comprom + $qtyToReserve,
                    ]);
                }
            }

            // ===== 3.3) Detalle (por el 100%)
            PosVentaDetalle::create([
                'pos_inventario_id'      => $posInvIdFisico,   // puede quedar null si TODO fue reserva o sin stock
                'pos_venta_id'           => $posVenta->id,
                'pos_bodega_articulo_id' => $articuloId ?: null,
                'pos_bodega_id'          => $bodegaId ?: null,
                'texto'                  => $request->observacion ?? '',
                'c_stock'                => $controlaStock,
                'cantidad'               => $cantidad,
                'unitario'               => $unitarioNeto,               // NETO unitario
                'neto'                   => round($netoItem),
                'impuesto'               => round($impuestoItem),
                'total'                  => round($totalItem),
            ]);
        }

        // ===== 4) Pagos
        foreach ((array) ($request->pagos ?? []) as $pago) {
                $cajaActiva = PosCaja::where('par_empresa_id', $request->empresa_id)
                ->where('par_sucursal_id', $request->sucursal_id)
                ->where('user_id', $request->usuario_id)
                ->where('estado', 1)
                ->first();

            if (!$cajaActiva) {
                throw new \Exception("No se encontró una caja activa para este usuario");
            }

            $cantidadPago = (int) ($pago['cantidad'] ?? 1);
            $montoUnit    = (float) ($pago['monto'] ?? 0);
            $montoFinal   = $montoUnit * max(1, $cantidadPago);

                PosCajaMovimiento::create([
                'fecha'                => $request->fecha,
                'motivo'               => 'Venta - ' . ($request->orden ?? $posVenta->id),
                'caja_id'              => $cajaActiva->id,
                'monto'                => round($montoFinal),
                'metodo_id'            => $pago['metodo_id'] ?? null,
                'caja_t_movimiento_id' => $pago['caja_t_movimiento_id'] ?? null,
                'par_empresa_id'       => $request->empresa_id,
                'par_sucursal_id'          => $request->sucursal_id,
            ]);
        }

        DB::commit();

        return response()->json([
            'success' => true,
            'venta'   => $posVenta,
            'message' => 'Venta guardada exitosamente',
        ]);

    } catch (\Exception $e) {
        DB::rollBack();
        return response()->json([
            'success' => false,
            'message' => 'Error al guardar la venta: ' . $e->getMessage(),
        ], 500);
    }
}






    /**
     * Display the specified resource.
     *
     * @param  \App\Models\PosVenta  $posVenta
     * @return \Illuminate\Http\Response
     */
    public function show(PosVenta $posVenta)
    {
        // Retornar la venta con sus relaciones y detalles
        $venta = PosVenta::with(['Documento', 'Metodo', 'Cliente', 'Detalles'])
                        ->where('id', $posVenta->id)
                        ->first();

        if (!$venta) {
            return null;
        }

        // Mapear detalles para incluir información útil si es necesario
        $venta->detalles = $venta->Detalles->map(function ($d) {
            return [
                'id' => $d->id,
                'pos_inventario_id' => $d->pos_inventario_id,
                'pos_bodega_articulo_id' => $d->pos_bodega_articulo_id,
                'pos_bodega_id' => $d->pos_bodega_id,
                'texto' => $d->texto,
                'c_stock' => $d->c_stock,
                'cantidad' => $d->cantidad,
                'unitario' => $d->unitario,
                'neto' => $d->neto,
                'impuesto' => $d->impuesto,
                'total' => $d->total,
            ];
        });

        return $venta;
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\PosVenta  $posVenta
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, PosVenta $posVenta)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\PosVenta  $posVenta
     * @return \Illuminate\Http\Response
     */
    public function destroy(PosVenta $posVenta)
    {
        //
    }
}
 