<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Arr;

class Cors
{
    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure(\Illuminate\Http\Request): (\Illuminate\Http\Response|\Illuminate\Http\RedirectResponse)  $next
     * @return \Illuminate\Http\Response|\Illuminate\Http\RedirectResponse
     */
    public function handle(Request $request, Closure $next)
    {
        // Load CORS config
        $corsConfig = config('cors', []);

        $allowedOrigins = Arr::wrap($corsConfig['allowed_origins'] ?? ['*']);
        $allowedMethods = Arr::wrap($corsConfig['allowed_methods'] ?? ['*']);
        $allowedHeaders = Arr::wrap($corsConfig['allowed_headers'] ?? ['*']);
        $supportsCredentials = $corsConfig['supports_credentials'] ?? false;

        $origin = $request->header('Origin');

        // Decide which origin to send back
        if (in_array('*', $allowedOrigins) || ($origin && in_array($origin, $allowedOrigins))) {
            $allowOrigin = in_array('*', $allowedOrigins) ? '*' : $origin;
        } else {
            // origin not allowed; do not add CORS headers
            $allowOrigin = null;
        }

        $allowMethods = in_array('*', $allowedMethods) ? 'POST, GET, OPTIONS, PUT, DELETE' : implode(', ', $allowedMethods);

        // Prefer the requested headers in preflight if present
        $requestHeaders = $request->header('Access-Control-Request-Headers');
        if ($requestHeaders) {
            $allowHeaders = $requestHeaders;
        } else {
            $allowHeaders = in_array('*', $allowedHeaders) ? '*' : implode(', ', $allowedHeaders);
        }

        // Handle OPTIONS preflight early
        if ($request->getMethod() === 'OPTIONS') {
            $resp = response('', 204);
            if ($allowOrigin) {
                $resp->headers->set('Access-Control-Allow-Origin', $allowOrigin);
                $resp->headers->set('Vary', 'Origin');
            }
            $resp->headers->set('Access-Control-Allow-Methods', $allowMethods);
            $resp->headers->set('Access-Control-Allow-Headers', $allowHeaders);
            if ($supportsCredentials) {
                $resp->headers->set('Access-Control-Allow-Credentials', 'true');
            }
            return $resp;
        }

        // Normal request: pass through and then attach headers if origin allowed
        $response = $next($request);
        if ($allowOrigin) {
            $response->headers->set('Access-Control-Allow-Origin', $allowOrigin);
            $response->headers->set('Vary', 'Origin');
        }
        $response->headers->set('Access-Control-Allow-Methods', $allowMethods);
        $response->headers->set('Access-Control-Allow-Headers', $allowHeaders);
        if ($supportsCredentials) {
            $response->headers->set('Access-Control-Allow-Credentials', 'true');
        }

        return $response;
    }
    
}
