<?php

namespace App\Libraries\LibreDTE;

use App\Models\Venta;
use App\Models\Caf;
use App\Models\DocTributario;
use DOMDocument;
use Exception;

class DteBuilder
{
    public function generarFactura($venta_id)
    {
        \Log::info("🚀 Iniciando generación de DTE para venta ID: {$venta_id}");

        $venta = Venta::with(['cliente', 'empresa', 'venta_inventarios'])->findOrFail($venta_id);

        if (!$venta->folio) {
            \Log::error("❌ La venta no tiene folio asignado.");
            throw new Exception('La venta no tiene folio asignado.');
        }

        $cafModel = Caf::where('empresa_id', $venta->empresa_id)
            ->where('tipo_dte', 33)
            ->where('folio_desde', '<=', $venta->folio)
            ->where('folio_hasta', '>=', $venta->folio)
            ->first();

        if (!$cafModel) {
            \Log::error("❌ No se encontró CAF válido para empresa {$venta->empresa_id} y folio {$venta->folio}");
            throw new Exception('No se encontró un CAF válido para este folio.');
        }

        $codigoDTE = $this->obtenerCodigoDTE($venta->doc_tributario_id);
        if ($codigoDTE !== 33) {
            \Log::error("❌ Tipo de DTE inválido: {$codigoDTE}. Solo se permite tipo 33.");
            throw new Exception("Solo se permite generar DTE tipo 33.");
        }

        \Log::info("✅ Generando estructura XML DTE tipo 33 para folio {$venta->folio}");

        $doc = new DOMDocument('1.0', 'ISO-8859-1');
        $doc->formatOutput = true;

        $dte = $doc->createElementNS("http://www.sii.cl/SiiDte", "DTE");
        $dte->setAttribute("version", "1.0");
        $dte->setAttribute("ID", "DTE_T{$codigoDTE}_F{$venta->folio}");

        $documento = $doc->createElementNS("http://www.sii.cl/SiiDte", "Documento");
        $documento->setAttribute("ID", "T{$codigoDTE}F{$venta->folio}");

        $encabezado = $doc->createElement("Encabezado");
        $idDoc = $doc->createElement("IdDoc");
        $idDoc->appendChild($doc->createElement("TipoDTE", $codigoDTE));
        $idDoc->appendChild($doc->createElement("Folio", $venta->folio));
        $idDoc->appendChild($doc->createElement("FchEmis", $venta->fecha));

        $emisor = $doc->createElement("Emisor");
        $emisor->appendChild($doc->createElement("RUTEmisor", $this->formatearRutConGuion($venta->empresa->rut)));
        $emisor->appendChild($doc->createElement("RznSoc", htmlspecialchars($venta->empresa->razon_social)));
        $emisor->appendChild($doc->createElement("GiroEmis", htmlspecialchars($venta->empresa->giro ?: 'Comercio')));
        $emisor->appendChild($doc->createElement("Acteco", $venta->empresa->acteco ?: '726000'));
        $emisor->appendChild($doc->createElement("DirOrigen", htmlspecialchars($venta->empresa->direccion)));
        $emisor->appendChild($doc->createElement("CmnaOrigen", substr(htmlspecialchars($venta->empresa->comuna), 0, 20)));

        $receptor = $doc->createElement("Receptor");
        $receptor->appendChild($doc->createElement("RUTRecep", $this->formatearRutConGuion($venta->cliente->rut)));
        $receptor->appendChild($doc->createElement("RznSocRecep", htmlspecialchars($venta->cliente->razon_social)));
        $receptor->appendChild($doc->createElement("GiroRecep", htmlspecialchars($venta->cliente->giro ?: 'Particular')));
        $receptor->appendChild($doc->createElement("DirRecep", htmlspecialchars($venta->cliente->direccion)));
        $receptor->appendChild($doc->createElement("CmnaRecep", substr(htmlspecialchars($venta->cliente->comuna), 0, 20)));

        $totales = $doc->createElement("Totales");
        $totales->appendChild($doc->createElement("MntNeto", round($venta->neto)));
        $totales->appendChild($doc->createElement("IVA", round($venta->impuesto)));
        $totales->appendChild($doc->createElement("MntTotal", round($venta->total)));

        $encabezado->appendChild($idDoc);
        $encabezado->appendChild($emisor);
        $encabezado->appendChild($receptor);
        $encabezado->appendChild($totales);

        $documento->appendChild($encabezado);

        foreach ($venta->venta_inventarios as $i => $item) {
            $detalle = $doc->createElement("Detalle");
            $detalle->appendChild($doc->createElement("NroLinDet", $i + 1));
            $detalle->appendChild($doc->createElement("NmbItem", htmlspecialchars($item->texto)));
            $detalle->appendChild($doc->createElement("QtyItem", number_format($item->cantidad, 2, '.', '')));
            $detalle->appendChild($doc->createElement("PrcItem", round($item->unitario)));
            $detalle->appendChild($doc->createElement("MontoItem", round($item->neto)));
            $documento->appendChild($detalle);
        }

        $ted = new TedGenerator();
        $nodoTED = $ted->generarTED($venta, $cafModel);
        $documento->appendChild($doc->importNode($nodoTED, true));
        $documento->appendChild($doc->createElement("TmstFirma", date('Y-m-d\TH:i:s')));

        $dte->appendChild($documento);
        $doc->appendChild($dte);

        $firma = new FirmaElectronica(storage_path('app/' . env('CERTIFICADO_PFX')), env('CERTIFICADO_PASSWORD'));
        $xmlFirmado = $firma->firmarXML($doc, 'DTE');

        $ruta = "dtes/empresa_{$venta->empresa_id}/";
        $nombreArchivo = "DTE_{$codigoDTE}_F{$venta->folio}.xml";
        $rutaCompleta = storage_path("app/" . $ruta);
        if (!file_exists($rutaCompleta)) {
            mkdir($rutaCompleta, 0777, true);
        }

        file_put_contents($rutaCompleta . $nombreArchivo, $xmlFirmado);
        \Log::info("✅ DTE guardado: {$ruta}{$nombreArchivo}");

        return $ruta . $nombreArchivo;
    }

    private function obtenerCodigoDTE($docTributarioId)
    {
        $documento = DocTributario::find($docTributarioId);
        if (!$documento || !$documento->codigo) {
            throw new Exception("❌ No se encontró código DTE válido para doc_tributario_id {$docTributarioId}.");
        }
        return (int) $documento->codigo;
    }

    private function formatearRutConGuion($rut)
    {
        return substr($rut, 0, -1) . '-' . substr($rut, -1);
    }
}