<?php

namespace App\Libraries\LibreDTE;

use App\Models\Venta;
use App\Models\Dte;
use DOMDocument;
use Exception;
use Illuminate\Support\Facades\Storage;
use SoapClient;

class EnvioSii
{
    public function enviar($ventaId)
    {
        \Log::info("🚀 Iniciando envío de DTE al SII para venta ID: {$ventaId}");

        $venta = Venta::with('empresa')->findOrFail($ventaId);
        $empresa = $venta->empresa;

        $rutaXmlDte = storage_path("app/dtes/empresa_{$empresa->id}/DTE_33_F{$venta->folio}.xml");

        if (!file_exists($rutaXmlDte)) {
            throw new Exception("❌ No se encontró el archivo XML del DTE: {$rutaXmlDte}");
        }

        \Log::info("📄 Cargando DTE desde archivo: {$rutaXmlDte}");
        $xmlDte = file_get_contents($rutaXmlDte);

        \Log::info("📦 Preparando estructura EnvioDTE...");
        $envio = new DOMDocument('1.0', 'ISO-8859-1');
        $envio->formatOutput = false;

        $envioDte = $envio->createElementNS("http://www.sii.cl/SiiDte", "EnvioDTE");
        $envioDte->setAttribute("version", "1.0");
        $envioDte->setAttributeNS(
            "http://www.w3.org/2001/XMLSchema-instance",
            "xsi:schemaLocation",
            "http://www.sii.cl/SiiDte EnvioDTE_v10.xsd"
        );

        $setDte = $envio->createElement("SetDTE");
        $setDte->setAttribute("ID", "SetDoc");

        \Log::info("🧾 Agregando carátula...");
        $caratula = $envio->createElement("Caratula");
        $caratula->setAttribute("version", "1.0");
        $caratula->appendChild($envio->createElement("RutEmisor", $this->formatearRutConGuion($empresa->rut)));
        $caratula->appendChild($envio->createElement("RutEnvia", env('RUT_ENVIA')));
        $caratula->appendChild($envio->createElement("RutReceptor", "60803000-K"));
        $caratula->appendChild($envio->createElement("FchResol", $empresa->fch_resol));
        $caratula->appendChild($envio->createElement("NroResol", $empresa->nro_resol));
        $caratula->appendChild($envio->createElement("TmstFirmaEnv", date('Y-m-d\TH:i:s')));
        $setDte->appendChild($caratula);

        \Log::info("📄 Importando nodo <Documento> al SetDTE...");
        $dteDoc = new DOMDocument();
        $dteDoc->loadXML($xmlDte);
        $documentoNode = $dteDoc->getElementsByTagName('Documento')->item(0);

        if (!$documentoNode) {
            throw new Exception("❌ No se encontró el nodo <Documento> en el XML.");
        }

        $importado = $envio->importNode($documentoNode, true);
        $setDte->appendChild($importado);

        $envioDte->appendChild($setDte);
        $envio->appendChild($envioDte);

        \Log::info("✍️ Firmando nodo <SetDTE>...");
        $firma = new FirmaElectronica(storage_path('app/' . env('CERTIFICADO_PFX')), env('CERTIFICADO_PASSWORD'));
        $xmlFirmado = $firma->firmarXML($envio->saveXML(), 'SetDTE');

        \Log::info("📥 Guardando XML firmado de EnvioDTE...");
        $nombreEnvio = 'envioDTE_' . now()->format('Ymd_His') . '.xml';
        $rutaRelativa = "dtes/empresa_{$empresa->id}/{$nombreEnvio}";
        Storage::put($rutaRelativa, $xmlFirmado);
        \Log::info("✅ XML firmado guardado en: {$rutaRelativa}");

        \Log::info("🔐 Solicitando token al SII...");
        $token = $firma->obtenerToken();
        \Log::info("🔑 Token recibido: {$token}");

        \Log::info("📤 Enviando EnvioDTE al endpoint: " . env('SII_ENVIO_URL'));
        $endpoint = env('SII_ENVIO_URL');
        $context = stream_context_create([
            'http' => [
                'header' => "Content-type: text/xml;charset=ISO-8859-1\r\nAuthorization: TOKEN $token"
            ]
        ]);

        $client = new SoapClient(null, [
            'location' => $endpoint,
            'uri' => 'http://tempuri.org/',
            'trace' => 1,
            'stream_context' => $context,
        ]);

        $response = $client->__doRequest($xmlFirmado, $endpoint, '', 1);
        $xmlResp = simplexml_load_string($response);

        if (!$xmlResp || !isset($xmlResp->RESPUESTA->TRACKID)) {
            \Log::error("❌ Respuesta inválida del SII:\n$response");
            throw new Exception("❌ Respuesta inválida del SII:\n" . $response);
        }

        $trackId = (string)$xmlResp->RESPUESTA->TRACKID;
        \Log::info("📬 Track ID recibido del SII: {$trackId}");

        Dte::where('venta_id', $venta->id)->update([
            'ruta_envio' => $rutaRelativa
        ]);

        \Log::info("📌 DTE actualizado con ruta de envío: {$rutaRelativa}");

        return $trackId;
    }

    private function formatearRutConGuion($rut)
    {
        return substr($rut, 0, -1) . '-' . substr($rut, -1);
    }
}
