<?php

namespace App\Libraries\LibreDTE;

use DOMDocument;
use Exception;
use RobRichards\XMLSecLibs\XMLSecurityDSig;
use RobRichards\XMLSecLibs\XMLSecurityKey;

class FirmaElectronica
{
    public $cert;
    public $pkey;
    public $data;
    public $password;

    public function __construct($certificado, $password)
    {
        \Log::info("🔐 Cargando certificado desde: {$certificado}");
        $this->data = file_get_contents($certificado);
        if (!$this->data) {
            \Log::error("❌ No se pudo leer el archivo del certificado PFX.");
            throw new Exception('No se pudo leer el certificado.');
        }

        if (!openssl_pkcs12_read($this->data, $certs, $password)) {
            \Log::error("❌ Falló la lectura del PFX: Clave incorrecta o certificado corrupto.");
            throw new Exception('No se pudo leer el .pfx o la clave es incorrecta.');
        }

        $this->cert = $certs['cert'];
        $this->pkey = $certs['pkey'];
        \Log::info("✅ Certificado y clave privada cargados correctamente.");
    }

    public function firmarXML($input, $tag)
    {
        \Log::info("✍️ Iniciando firma XML sobre el nodo <{$tag}>...");
    
        $doc = ($input instanceof \DOMDocument) ? $input : new \DOMDocument();
        if (!$input instanceof \DOMDocument) {
            $doc->preserveWhiteSpace = false;
            $doc->formatOutput = false;
            $doc->loadXML($input);
        }
    
        $nodo = $doc->getElementsByTagName($tag)->item(0);
        if (!$nodo) {
            \Log::error("❌ No se encontró el nodo <{$tag}> en el XML.");
            throw new \Exception("El nodo '$tag' no se encontró en el XML.");
        }
    
        $idFirmado = $nodo->getAttribute('ID');
        if (!$idFirmado) {
            \Log::error("❌ El nodo <{$tag}> no tiene atributo ID.");
            throw new \Exception("El nodo '$tag' no contiene atributo ID para firma.");
        }
    
        $objDSig = new XMLSecurityDSig();
        $objDSig->setCanonicalMethod(XMLSecurityDSig::C14N);
        $objDSig->addReference(
            $nodo,
            XMLSecurityDSig::SHA1,
            ['http://www.w3.org/2000/09/xmldsig#enveloped-signature'],
            ['uri' => '#' . $idFirmado]
        );
    
        $objKey = new XMLSecurityKey(XMLSecurityKey::RSA_SHA1, ['type' => 'private']);
        $objKey->loadKey($this->pkey, false);
    
        $objDSig->sign($objKey);
        $objDSig->add509Cert($this->cert);
    
        // ✅ Insertar la firma como hijo de <DTE>, no de <Documento>
        if ($tag === 'DTE') {
            $doc->documentElement->appendChild($objDSig->sigNode);
        } else {
            $objDSig->appendSignature($nodo);
        }
    
        \Log::info("✅ Firma XML completada exitosamente.");
        return $doc->saveXML();
    }
    
    

    public function firmarSemilla($semilla)
    {
        \Log::info("✍️ Firmando semilla: {$semilla}");

        $xml = '<?xml version="1.0" encoding="UTF-8"?>';
        $xml .= '<getSeed xmlns="http://www.sii.cl/SiiDte">';
        $xml .= '<Semilla>' . $semilla . '</Semilla>';
        $xml .= '</getSeed>';

        $doc = new DOMDocument();
        $doc->preserveWhiteSpace = false;
        $doc->formatOutput = false;
        $doc->loadXML($xml);

        $objDSig = new XMLSecurityDSig();
        $objDSig->setCanonicalMethod(XMLSecurityDSig::EXC_C14N);
        $objDSig->addReference(
            $doc,
            XMLSecurityDSig::SHA1,
            ['http://www.w3.org/2000/09/xmldsig#enveloped-signature']
        );

        $objKey = new XMLSecurityKey(XMLSecurityKey::RSA_SHA1, ['type' => 'private']);
        $objKey->loadKey($this->pkey, false);

        $objDSig->sign($objKey);
        $objDSig->add509Cert($this->cert);
        $objDSig->appendSignature($doc->documentElement);

        \Log::info("✅ Semilla firmada correctamente.");
        return $doc->saveXML();
    }

    public function generarFirma($data)
    {
        \Log::info("🔏 Generando firma binaria para TED...");
        if (!openssl_sign($data, $firmaBinaria, $this->pkey, OPENSSL_ALGO_SHA1)) {
            \Log::error("❌ No se pudo firmar el nodo DD.");
            throw new Exception("No se pudo firmar el nodo DD.");
        }

        \Log::info("✅ Firma binaria generada correctamente.");
        return $firmaBinaria;
    }

    public function obtenerToken()
    {
        \Log::info("🚀 Iniciando solicitud de token al SII...");

        // Paso 1: Obtener semilla
        $xmlSeed = '<getSeed xmlns="http://www.sii.cl/SiiDte"></getSeed>';
        $soapRequest = <<<XML
<?xml version="1.0" encoding="UTF-8"?>
<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/" xmlns:sii="http://www.sii.cl/SiiDte">
    <soapenv:Header/>
    <soapenv:Body>{$xmlSeed}</soapenv:Body>
</soapenv:Envelope>
XML;

        $seedUrl = env('SII_CR_SEED_URL');
        \Log::info("📡 Enviando solicitud de semilla a: {$seedUrl}");
        $ch = curl_init($seedUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $soapRequest);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: text/xml; charset=utf-8',
            'SOAPAction: ""'
        ]);
        $response = curl_exec($ch);
        curl_close($ch);

        preg_match('/<getSeedReturn>(.*?)<\/getSeedReturn>/s', $response, $match);
        if (!isset($match[1])) {
            \Log::error("❌ No se pudo obtener la semilla del SII.");
            throw new \Exception("No se pudo obtener la semilla del SII.");
        }

        preg_match('/<SEMILLA>(.*?)<\/SEMILLA>/', html_entity_decode($match[1]), $semillaMatch);
        $semilla = $semillaMatch[1] ?? null;

        if (!$semilla) {
            \Log::error("❌ No se pudo extraer la SEMILLA del XML.");
            throw new \Exception("No se pudo extraer la SEMILLA.");
        }

        \Log::info("✅ Semilla obtenida: {$semilla}");

        // Paso 2: Firmar semilla
        $xmlFirmado = $this->firmarSemilla($semilla);

        // Paso 3: Obtener token
        $soapToken = <<<XML
<?xml version="1.0" encoding="UTF-8"?>
<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/" xmlns:sii="http://www.sii.cl/SiiDte">
    <soapenv:Header/>
    <soapenv:Body>
        <getToken>
            <SignedSeed><![CDATA[{$xmlFirmado}]]></SignedSeed>
        </getToken>
    </soapenv:Body>
</soapenv:Envelope>
XML;

        $tokenUrl = env('SII_GET_TOKEN_URL');
        \Log::info("📡 Solicitando token desde: {$tokenUrl}");
        $ch = curl_init($tokenUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $soapToken);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: text/xml; charset=utf-8',
            'SOAPAction: ""'
        ]);
        $responseToken = curl_exec($ch);
        curl_close($ch);

        preg_match('/<TOKEN>(.*?)<\/TOKEN>/', $responseToken, $tokenMatch);
        $token = $tokenMatch[1] ?? null;

        if (!$token) {
            \Log::error("❌ No se pudo obtener el TOKEN del SII.");
            throw new \Exception("No se pudo obtener el TOKEN del SII.");
        }

        \Log::info("✅ TOKEN obtenido correctamente: {$token}");
        return $token;
    }
}
