<?php

namespace App\Libraries\LibreDTE;

use App\Models\Venta;
use App\Models\Caf;
use DOMDocument;
use Exception;

class TedGenerator
{
    public function generarTED(Venta $venta, Caf $caf)
    {
        \Log::info("🧾 Generando TED para venta ID: {$venta->id}, folio: {$venta->folio}");

        $doc = new DOMDocument('1.0', 'ISO-8859-1');
        $doc->formatOutput = true;

        $TED = $doc->createElement("TED");
        $TED->setAttribute("version", "1.0");

        $DD = $doc->createElement("DD");
        $DD->appendChild($doc->createElement("RE", $this->formatearRutConGuion($venta->empresa->rut)));
        $DD->appendChild($doc->createElement("TD", 33));
        $DD->appendChild($doc->createElement("F", $venta->folio));
        $DD->appendChild($doc->createElement("FE", $venta->fecha));
        $DD->appendChild($doc->createElement("RR", $this->formatearRutConGuion($venta->cliente->rut)));
        $DD->appendChild($doc->createElement("RSR", substr($venta->cliente->razon_social, 0, 40)));
        $DD->appendChild($doc->createElement("MNT", round($venta->total)));
        $DD->appendChild($doc->createElement("IT1", substr($venta->venta_inventarios->first()->texto ?? 'Venta', 0, 40)));

        // Añadir nodo CAF completo (incluyendo FRMA)
        try {
            $cafNode = $this->extraerCAF($doc, $caf);
            $DD->appendChild($cafNode);
            \Log::info("📌 Nodo CAF + FRMA añadido correctamente al TED.");
        } catch (Exception $e) {
            \Log::error("❌ Error al procesar CAF: " . $e->getMessage());
            throw $e;
        }

        $DD->appendChild($doc->createElement("TSTED", date('Y-m-d\TH:i:s')));
        $TED->appendChild($DD);

        // Firma nodo DD
        try {
            \Log::info("✍️ Firmando nodo DD del TED...");
            $firma = new FirmaElectronica(storage_path('app/' . env('CERTIFICADO_PFX')), env('CERTIFICADO_PASSWORD'));
            $frmt = $doc->createElement("FRMT", base64_encode($firma->generarFirma($DD->C14N())));
            $frmt->setAttribute("algoritmo", "SHA1withRSA");
            $TED->appendChild($frmt);
            \Log::info("✅ TED firmado exitosamente.");
        } catch (Exception $e) {
            \Log::error("❌ Error al firmar el TED: " . $e->getMessage());
            throw $e;
        }

        return $TED;
    }

    private function extraerCAF(DOMDocument $doc, Caf $caf)
    {
        \Log::info("📄 Cargando CAF XML...");

        $cafXml = new DOMDocument();
        $cafXml->loadXML($caf->caf_xml);

        $daNode = $cafXml->getElementsByTagName('DA')->item(0);
        $frmaNode = $cafXml->getElementsByTagName('FRMA')->item(0);

        if (!$daNode || !$frmaNode) {
            throw new Exception("El CAF no contiene nodos DA o FRMA válidos.");
        }

        $cafNode = $doc->createElement("CAF");
        $cafNode->setAttribute("version", "1.0");

        $cafNode->appendChild($doc->importNode($daNode, true));
        $cafNode->appendChild($doc->importNode($frmaNode, true));

        return $cafNode;
    }

    private function formatearRutConGuion($rut)
    {
        return substr($rut, 0, -1) . '-' . substr($rut, -1);
    }
}
