<?php

namespace App\Libraries\LibreDTE;

use DOMDocument;
use Exception;

class ValidadorXml
{
    /**
     * Valida un XML contra un archivo XSD.
     *
     * @param string $rutaXml Ruta del archivo XML.
     * @param string $rutaXsd Ruta del esquema XSD.
     * @return bool
     * @throws Exception
     */
    public function validar(string $rutaXml, string $rutaXsd): bool
    {
        if (!file_exists($rutaXml)) {
            throw new Exception("No se encontró el archivo XML en: $rutaXml");
        }

        if (!file_exists($rutaXsd)) {
            throw new Exception("No se encontró el archivo XSD en: $rutaXsd");
        }

        $doc = new DOMDocument();
        $doc->preserveWhiteSpace = false;
        $doc->formatOutput = false;

        if (!$doc->load($rutaXml)) {
            throw new Exception("No se pudo cargar el XML desde: $rutaXml");
        }

        libxml_use_internal_errors(true);

        if (!$doc->schemaValidate($rutaXsd)) {
            $errores = libxml_get_errors();
            libxml_clear_errors();

            $mensajes = array_map(function ($error) {
                return "[Línea {$error->line}] {$error->message}";
            }, $errores);

            throw new Exception("❌ Errores de validación contra XSD:\n" . implode("\n", $mensajes));
        }

        return true;
    }

    /**
     * Valida un DTE individual contra su esquema oficial.
     *
     * @param string $rutaXml Ruta al archivo XML del DTE.
     * @return bool
     * @throws Exception
     */
    public function validarDTE(string $rutaXml): bool
    {
        $xsdDte = base_path('resources/schemas/DTE_v10.xsd');
        return $this->validar($rutaXml, $xsdDte);
    }

    /**
     * Valida un EnvioDTE completo contra el esquema oficial.
     *
     * @param string $rutaXml Ruta al archivo XML del EnvioDTE.
     * @return bool
     * @throws Exception
     */
    public function validarEnvioDTE(string $rutaXml): bool
    {
        $xsdEnvio = base_path('resources/schemas/EnvioDTE_v10.xsd');
        return $this->validar($rutaXml, $xsdEnvio);
    }
}
