<?php

namespace App\Models\Pos;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PosArticulo extends Model
{
    use HasFactory;

    public const TIPO_SIMPLE   = 'simple';
    public const TIPO_VARIANTE = 'variante';
    public const TIPO_PACK     = 'pack';

    protected $table = 'pos_articulos';

    /**
     * Campos asignables
     */
    protected $fillable = [
        'nombre', 'barra', 'par_medida_id', 'par_marca_id', 'categoria_id',
        'venta', 'peso', 'stock_minimo', 'empresa_id', 'estado', 'c_stock',
        // nuevas para variantes/packs
        'tipo', 'parent_articulo_id', 'sku', 'atributos_json', 'atributos_hash',
    ];

    /**
     * Casts
     */
    protected $casts = [
        'atributos_json' => 'array', // JSON <-> array
    ];

    /**
     * Relaciones existentes
     */
    public function Empresa()   { return $this->belongsTo(\App\Models\Parametros\ParEmpresa::class); }
    public function Item()      { return $this->belongsTo(\App\Models\Item::class); }
    public function Marca()     { return $this->belongsTo(\App\Models\Parametros\ParMarca::class); }
    public function Medida()    { return $this->belongsTo(\App\Models\Parametros\ParMedida::class); }
    public function Categoria() { return $this->belongsTo(PosCategoria::class); }
    public function PosBodega() { return $this->belongsTo(PosBodega::class); }

    // OJO: tienes dos métodos con el mismo hasMany (nombres distintos), los dejo tal cual:
    public function PosInventarios() { return $this->hasMany(PosInventario::class); }
    public function PosInventario()  { return $this->hasMany(PosInventario::class); }



    /**
     * NUEVAS relaciones para variantes/packs
     */
    public function Padre()     { return $this->belongsTo(self::class, 'parent_articulo_id'); }

    // Stock por bodega del artículo (útil para variantes)
    public function BodegaArticulos()
    {
        return $this->hasMany(\App\Models\Pos\PosBodegaArticulo::class, 'pos_articulo_id');
    }

    // Si usas packs, deja esta relación (tabla pos_articulo_pack_items)
    public function PackItems()
    {
        return $this->hasMany(\App\Models\Pos\PosArticuloPackItem::class, 'pack_id');
    }

    // Alias y nombres amigables para el API (frontend espera 'impuestos', 'pack_items', 'variantes')
    public function impuestos()
    {
        return $this->hasMany(\App\Models\Pos\PosArticuloImpuesto::class, 'pos_articulo_id');
    }

    public function pack_items()
    {
        return $this->hasMany(\App\Models\Pos\PosArticuloPackItem::class, 'pack_id');
    }

    public function variantes()
    {
        return $this->hasMany(self::class, 'parent_articulo_id')
            ->where('tipo', self::TIPO_VARIANTE)
            ->orderBy('nombre');
    }

    /**
     * Scopes útiles
     */
    public function scopeVendibles($q)
    {
        return $q->whereIn('tipo', [self::TIPO_SIMPLE, self::TIPO_VARIANTE, self::TIPO_PACK]);
    }

    /**
     * Helpers
     */
    public function isVariante(): bool { return $this->tipo === self::TIPO_VARIANTE; }
    public function isPack(): bool     { return $this->tipo === self::TIPO_PACK; }

    // Si más adelante quieres heredar precio del padre, ajusta aquí.
    public function getPrecioFinalAttribute()
    {
        return $this->venta;
    }

    /**
     * Normaliza atributos y calcula hash automáticamente al guardar
     */
    protected static function booted()
    {
        static::saving(function (self $model) {
            // coherencia de parent_id
            if ($model->tipo !== self::TIPO_VARIANTE) {
                $model->parent_id = null;
            }

            // atributos_json -> array ordenado + hash
            if (!is_null($model->atributos_json)) {
                $attrs = is_array($model->atributos_json)
                    ? $model->atributos_json
                    : (json_decode($model->atributos_json, true) ?? []);

                // elimina vacíos
                $attrs = array_filter($attrs, fn ($v) => $v !== null && $v !== '');

                if (!empty($attrs)) {
                    ksort($attrs);
                    $firma = collect($attrs)
                        ->map(fn ($v, $k) => $k.'='.$v)
                        ->implode('|');

                    // re-asigna como array (Laravel lo casteará a JSON)
                    $model->atributos_json = $attrs;
                    $model->atributos_hash = hash('sha1', $firma);
                } else {
                    $model->atributos_json = null;
                    $model->atributos_hash = null;
                }
            } else {
                $model->atributos_hash = null;
            }
        });
    }
}
