<?php

namespace App\Models\Pos;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PosBodegaArticulo extends Model
{
    use HasFactory;

    protected $table = 'pos_bodega_articulos';

    protected $fillable = [
        'pos_articulo_id',
        'pos_bodega_id',
        'parent_bodega_articulo_id', // <<-- IMPORTANTE para poder setear el padre
        'c_stock',
        'stock',
        'en_origen_in',
        'p_venta',
        'costo',
        'c_unitario',
        'estado',
    'par_empresa_id',
    'par_sucursal_id',
        'en_transito_in',
        'comprometido',
    ];

    protected $casts = [
        // ids / enteros
        'id'                         => 'int',
        'pos_articulo_id'            => 'int',
        'pos_bodega_id'              => 'int',
        'parent_bodega_articulo_id'  => 'int',
    'par_empresa_id'                 => 'int',
    'par_sucursal_id'                => 'int',
        'estado'                     => 'int',
        'c_stock'                    => 'int',   // en la tabla es int(11)

        // decimales
        'stock'          => 'float', // decimal(18,3)
        'en_transito_in' => 'float', // decimal(18,3)
        'comprometido'   => 'float', // decimal(18,3)
        'p_venta'        => 'float', // decimal(12,2)
        'costo'          => 'float', // decimal(12,2)
        'c_unitario'     => 'float', // decimal(12,4)
        'en_origen_in'   => 'float', // decimal(18,4) en tu estructura
    ];

    // Exponer automáticamente el "disponible" en JSON
    protected $appends = ['disponible'];

    /** ================== Relaciones ================== */
    public function articulo()
    {
        return $this->belongsTo(PosArticulo::class, 'pos_articulo_id');
    }

    public function bodega()
    {
        return $this->belongsTo(PosBodega::class, 'pos_bodega_id');
    }

    // Relación al BA padre (solo aplica para variantes o hijos de pack, si decides usarlo)
    public function parent()
    {
        return $this->belongsTo(self::class, 'parent_bodega_articulo_id');
    }

    // Hijos (variantes o componentes que referencien a este BA como padre)
    public function children()
    {
        return $this->hasMany(self::class, 'parent_bodega_articulo_id');
    }

    /** ================== Atributos calculados ================== */
    public function getDisponibleAttribute(): float
    {
        $stock = (float) ($this->stock ?? 0);
        $comp  = (float) ($this->comprometido ?? 0);
        return max($stock - $comp, 0.0);
    }

    /** ================== Scopes prácticos ================== */
    public function scopeArticulo($q, int $articuloId)
    {
        return $q->where('pos_articulo_id', $articuloId);
    }

    public function scopeBodega($q, int $bodegaId)
    {
        return $q->where('pos_bodega_id', $bodegaId);
    }

    public function scopeEmpresa($q, int $empresaId)
    {
        return $q->where('empresa_id', $empresaId);
    }

    public function scopeOf($q, int $articuloId, int $bodegaId)
    {
        return $q->where('pos_articulo_id', $articuloId)
                 ->where('pos_bodega_id', $bodegaId);
    }

    // Solo los BA "raíz" (sin padre)
    public function scopeRoots($q)
    {
        return $q->whereNull('parent_bodega_articulo_id');
    }

    // Hijos de un BA específico
    public function scopeChildrenOf($q, int $parentBaId)
    {
        return $q->where('parent_bodega_articulo_id', $parentBaId);
    }

    /** ================== Defaults suaves al crear ================== */
    protected static function booted()
    {
        static::creating(function (self $m) {
            // defaults numéricos a 0 si vienen null
            foreach (['stock','c_stock','en_transito_in','comprometido','p_venta','costo','c_unitario','en_origen_in'] as $f) {
                if (is_null($m->{$f})) $m->{$f} = 0;
            }
            if (is_null($m->estado)) $m->estado = 1;
        });
    }

    /** ================== Helpers mínimos ================== */
    public function incrementarStock(float $cantidad): void
    {
        $this->increment('stock', $cantidad);
    }

    public function decrementarStock(float $cantidad): void
    {
        $this->decrement('stock', $cantidad);
    }
}
