<?php

namespace App\Services;

use App\Models\Pos\PosInventario;
use App\Models\Pos\PosBodegaArticulo;

class InventoryCostService
{
    public function costearFIFO(int $articuloId, int $bodegaId, float $cantidad): float
    {
        return $this->costear($articuloId, $bodegaId, $cantidad, 'asc');
    }

    public function costearLIFO(int $articuloId, int $bodegaId, float $cantidad): float
    {
        return $this->costear($articuloId, $bodegaId, $cantidad, 'desc');
    }

    /**
     * Costeo por consumo de lotes:
     * ACEPTA mov=1 (Entrada) y mov=6 (Recepcionado de traspaso), ambos con c_stock=1 y estado=1.
     * $articuloId es pos_bodega_articulo_id (PBA).
     */
    private function costear(int $articuloId, int $bodegaId, float $cantidad, string $orden): float
    {
        if ($cantidad <= 0) {
            throw new \Exception("La cantidad a costear debe ser mayor que cero.");
        }

        $lotes = PosInventario::where('pos_bodega_articulo_id', $articuloId)
            ->where('pos_bodega_id', $bodegaId)
            ->whereIn('movimiento', [1, 6])
            ->where('estado', 1)
            ->where('c_stock', 1)
            ->orderBy('fecha', $orden)
            ->orderBy('id', $orden)
            ->lockForUpdate()
            ->get();

        $restante = $cantidad;
        $costoTot = 0.0;

        foreach ($lotes as $lote) {
            $cantLote   = (float) ($lote->cantidad ?? 0);
            $consumido  = (float) ($lote->consumido ?? 0);
            $disponible = max(0.0, $cantLote - $consumido);
            if ($disponible <= 0) continue;

            $usar = min($disponible, $restante);
            $cu   = (float) ($lote->compra ?? 0);
            $costoTot += $usar * $cu;

            $lote->consumido = $consumido + $usar;
            $lote->save();

            $restante -= $usar;
            if ($restante <= 0) break;
        }

        if ($restante > 0) {
            throw new \Exception("Stock insuficiente para el artículo {$articuloId} en bodega {$bodegaId}.");
        }

        return $costoTot / $cantidad;
    }

    /**
     * Identificación específica:
     * Permite mov=1 o mov=6 (c_stock=1, estado=1).
     */
    public function costearIdentificacion(int $loteId, float $cantidad, ?int $articuloId = null, ?int $bodegaId = null): float
    {
        if ($cantidad <= 0) {
            throw new \Exception("La cantidad a costear debe ser mayor que cero.");
        }

        $q = PosInventario::where('id', $loteId)
            ->whereIn('movimiento', [1, 6])
            ->where('estado', 1)
            ->where('c_stock', 1)
            ->lockForUpdate();

        if ($articuloId !== null) $q->where('pos_bodega_articulo_id', $articuloId);
        if ($bodegaId !== null)   $q->where('pos_bodega_id', $bodegaId);

        $lote = $q->firstOrFail();

        $consumido  = (float) ($lote->consumido ?? 0);
        $disponible = max(0.0, (float)$lote->cantidad - $consumido);
        if ($disponible < $cantidad) {
            throw new \Exception("El lote {$loteId} no tiene saldo suficiente. Disponible: {$disponible}, requerido: {$cantidad}.");
        }

        $lote->consumido = $consumido + $cantidad;
        $lote->save();

        return (float) ($lote->compra ?? 0);
    }

    /**
     * Estimación de costo:
     * Prioriza mov=6 → mov=1 → mov=4 → c_unitario PBA → promedio → última salida → 0.0
     */
    public function estimarCostoUnitario(int $articuloId, int $bodegaId): float
    {
        $ult = PosInventario::where('pos_bodega_articulo_id', $articuloId)
            ->where('pos_bodega_id', $bodegaId)
            ->whereIn('movimiento', [6, 1])
            ->where('compra', '>', 0)
            ->orderByDesc('fecha')->orderByDesc('id')->first();
        if ($ult) return (float)$ult->compra;

        $ult4 = PosInventario::where('pos_bodega_articulo_id', $articuloId)
            ->where('pos_bodega_id', $bodegaId)
            ->where('movimiento', 4)
            ->where('compra', '>', 0)
            ->orderByDesc('fecha')->orderByDesc('id')->first();
        if ($ult4) return (float)$ult4->compra;

        $ba = PosBodegaArticulo::find($articuloId);
        if ($ba && (float)$ba->c_unitario > 0) return (float)$ba->c_unitario;

        $avg = PosInventario::where('pos_bodega_articulo_id', $articuloId)
            ->where('pos_bodega_id', $bodegaId)
            ->whereIn('movimiento', [6, 1])
            ->where('compra', '>', 0)
            ->selectRaw('SUM(compra * cantidad) / NULLIF(SUM(cantidad),0) as cu_prom')
            ->value('cu_prom');
        if ($avg && $avg > 0) return (float)$avg;

        $ultOut = PosInventario::where('pos_bodega_articulo_id', $articuloId)
            ->where('pos_bodega_id', $bodegaId)
            ->where('movimiento', 2)
            ->where('compra', '>', 0)
            ->orderByDesc('fecha')->orderByDesc('id')->first();
        if ($ultOut) return (float)$ultOut->compra;

        return 0.0;
    }
}
